package net.bluemind.delivery.smtp.ndr;

import java.util.List;
import java.util.Optional;

import org.columba.ristretto.smtp.SMTPException;

public class FailedRecipient {

	public final String email;
	public final int code;
	public final String message;
	public final String smtpStatus;

	public FailedRecipient(int code, String email, String message, String smtpStatus) {
		this.code = code;
		this.email = email;
		this.message = message;
		this.smtpStatus = smtpStatus;
	}

	private static FailedRecipient create(int code, String email, String message) {
		if (message == null || message.isEmpty()) {
			return new FailedRecipient(code, email, "",
					SendmailHelper.isSuccessCode(code) ? SendmailHelper.SMTP_OK_STATUS
							: SendmailHelper.SMTP_ERROR_STATUS);
		}
		Optional<String> status = SendmailHelper.getSmtpErrorCode(code, message);
		if (status.isPresent()) {
			message = message.substring(status.get().length());
		}

		return new FailedRecipient(code, email, message, status.orElse(SendmailHelper.SMTP_ERROR_STATUS));
	}

	public boolean isError() {
		return SendmailHelper.isError(code);
	}

	public boolean isSuccess() {
		return SendmailHelper.isSuccessCode(code);
	}

	public String action() {
		return isSuccess() ? "delivered" : "failed";
	}

	public static FailedRecipient create(String code, String mailAddress, String message) {
		return FailedRecipient.create(Integer.valueOf(code), mailAddress, message);
	}

	public static FailedRecipient ok(String mailAddress) {
		return FailedRecipient.create(200, mailAddress, "");
	}

	public static FailedRecipient failure(Exception ex, String mailAddress) {
		int code = 500;
		if (ex instanceof SMTPException sme) {
			code = sme.getCode();
		}
		return FailedRecipient.create(code, mailAddress, ex.getMessage());
	}

	@Override
	public String toString() {
		return String.format("%d %s (%s) - %s", code, email, message, smtpStatus);
	}

	public static void addErrorFailedRecipients(List<FailedRecipient> failedRecipients, Exception ex,
			List<String> recipients) {
		failedRecipients.removeIf(fr -> fr.isSuccess());
		List<String> mailInErrors = failedRecipients.stream().filter(fr -> fr.isError()).map(fr -> fr.email).toList();
		failedRecipients.addAll(recipients.stream().filter(r -> !mailInErrors.contains(r))
				.map(r -> FailedRecipient.failure(ex, r)).toList());
	}

}
