package net.bluemind.exchange.mapi.persistence.pclcache.migration;

import java.util.Objects;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.sql.DataSource;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.exchange.mapi.api.MapiPCLCacheEntry;
import net.bluemind.exchange.mapi.persistence.pclcache.MapiPCLCacheStore;
import net.bluemind.exchange.mapi.repository.IMapiPCLCacheStore;

/**
 * This store is used to manage pcl migration from mapdb to postgres we always
 * use postgres to write during migration we first try to read result from
 * postgres, if nothing is found we return result from mapDb
 */
public class MapiPCLCachePostgresToMapDbMigrationStore implements IMapiPCLCacheStore {

	private MapiPCLCacheStore postgresStore;
	private MapiPCLCacheMapDbStore mapDbStore;

	public MapiPCLCachePostgresToMapDbMigrationStore(DataSource dataSource) {
		postgresStore = new MapiPCLCacheStore(dataSource);
		mapDbStore = MapiPCLCacheMapDbStore.get();
	}

	@Override
	public void store(UUID replicaGuid, Long globalCounter, MapiPCLCacheEntry entry) throws ServerFault {
		writeInStore(replicaGuid, store -> store.store(replicaGuid, globalCounter, entry));
	}

	@Override
	public MapiPCLCacheEntry get(UUID replicaGuid, Long globalCounter) throws ServerFault {
		return readWithMapDbFallback(replicaGuid, Objects::isNull, store -> store.get(replicaGuid, globalCounter));
	}

	@Override
	public boolean contains(UUID replicaGuid, Long globalCounter) throws ServerFault {
		return readWithMapDbFallback(replicaGuid, result -> !result,
				store -> store.contains(replicaGuid, globalCounter));
	}

	@Override
	public void delete(UUID replicaGuid, Long globalCounter) {
		writeInStore(replicaGuid, store -> store.delete(replicaGuid, globalCounter));

	}

	private void writeInStore(UUID replicaGuid, Consumer<IMapiPCLCacheStore> exec) {
		boolean userMigrated = PclCacheMigrationDoneFlag.get().isDone(replicaGuid)
				|| PclCacheMigrationDoneFlag.get().isAllDone();
		if (userMigrated) {
			exec.accept(postgresStore);
		} else {
			exec.accept(postgresStore);
			exec.accept(mapDbStore);
		}
	}

	private <T> T readWithMapDbFallback(UUID replicaGuid, Predicate<T> shouldFallback,
			Function<IMapiPCLCacheStore, T> exec) {
		boolean userMigrated = PclCacheMigrationDoneFlag.get().isDone(replicaGuid)
				|| PclCacheMigrationDoneFlag.get().isAllDone();
		T coreResult = null;
		// if user is migrated call core store
		// otherwise try to find result in core store,
		// if result not found try to find in mapdb store
		if (userMigrated) {
			coreResult = exec.apply(postgresStore);
		} else {
			coreResult = exec.apply(postgresStore);
			if (shouldFallback.test(coreResult)) {
				coreResult = exec.apply(mapDbStore);
			}
		}
		return coreResult;
	}

}
