/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.dav.server.proto.put;

import java.net.URLDecoder;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.http.HttpServerResponse;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.dav.server.proto.DavHeaders;
import net.bluemind.dav.server.proto.IDavProtocol;
import net.bluemind.dav.server.store.DavResource;
import net.bluemind.dav.server.store.DavStore;
import net.bluemind.dav.server.store.LoggedCore;
import net.bluemind.vertx.common.Body;

public class PutProtocol implements IDavProtocol<PutQuery, PutResponse> {

	private static final Logger logger = LoggerFactory.getLogger(PutProtocol.class);

	private static final Pattern uidPattern = Pattern.compile("(.+?/calendars/__uids__/[^/]+/[^/]+/)(.+)(.ics|.vcf)");

	@Override
	public void parse(final HttpServerRequest r, final DavResource davRes, final Handler<PutQuery> handler) {
		Body.handle(r, new Handler<Buffer>() {

			@Override
			public void handle(Buffer buffer) {
				for (String hn : r.headers().names()) {
					if (!"authorization".equals(hn)) {
						logger.info("{}: {}", hn, r.headers().get(hn));
					}
				}

				String contentType = r.headers().get("Content-Type");
				PutQuery putQuery = PutQuery.of(davRes, contentType, buffer);

				String path = r.path();
				logger.info("[{}] parse {}\n{}", buffer.length(), path, buffer);

				DavHeaders.parse(putQuery, r.headers());
				putQuery.setCreate(r.headers().contains("If-None-Match"));

				Matcher uidMatcher = uidPattern.matcher(path);
				if (isUidPut(uidMatcher)) {
					parseUidPutPath(putQuery, uidMatcher);
				} else {
					parseGenericPut(putQuery, path);
				}

				handler.handle(putQuery);
			}

			private void parseGenericPut(PutQuery putQuery, String path) {
				int lastDot = path.lastIndexOf('.');
				int lastSlash = path.lastIndexOf('/', lastDot);
				if (lastDot > 0 && lastSlash > 0) {
					try {
						String col = path.substring(0, lastSlash + 1);
						String extId = URLDecoder.decode(path.substring(lastSlash + 1, lastDot), "UTF-8");
						putQuery.setCollection(col);
						putQuery.setExtId(extId);
					} catch (Exception e) {
						logger.error(e.getMessage(), e);
					}
				}
			}

			private void parseUidPutPath(PutQuery putQuery, Matcher uidMatcher) {
				try {
					String col = uidMatcher.group(1);
					String extId = URLDecoder.decode(uidMatcher.group(2), "UTF-8");
					putQuery.setCollection(col);
					putQuery.setExtId(extId);
				} catch (Exception e) {
					logger.error(e.getMessage(), e);
				}
			}

			private boolean isUidPut(Matcher uidMatcher) {
				return uidMatcher.matches();
			}

		});
	}

	@Override
	public void execute(LoggedCore lc, PutQuery query, Handler<PutResponse> handler) {
		logger.info("execute");

		PutResponse pr = new PutResponse();
		pr.setStatus(501);

		String col = query.getCollection();
		DavStore ds = new DavStore(lc);
		DavResource dr = ds.from(col);
		ContainerDescriptor cal = lc.vStuffContainer(dr);
		try {
			CreateEntity.getByType(cal.type).create(lc, query, pr, cal);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			pr.setStatus(500);
		}

		handler.handle(pr);
	}

	@Override
	public void write(PutResponse response, HttpServerResponse sr) {
		if (response.getEtag() != null) {
			sr.headers().set("Etag", response.getEtag());
		}
		sr.setStatusCode(response.getStatus()).end();
	}

}
