package net.bluemind.central.reverse.proxy.model.common.kafka.impl;

import static java.time.Duration.ofSeconds;

import java.util.List;
import java.util.Objects;
import java.util.Properties;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.RateLimiter;

import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.Promise;
import net.bluemind.central.reverse.proxy.model.common.kafka.KafkaConsumerClient;
import net.bluemind.central.reverse.proxy.model.common.kafka.LagConsumer;

public class KafkaConsumerClientImpl<K, V> implements KafkaConsumerClient<K, V> {

	private final Logger logger = LoggerFactory.getLogger(KafkaConsumerClientImpl.class);
	private static final int MIN_UPTIME_BEFORE_EMPTY_CONSUMPTION = 10;
	private static final int POLL_DURATION_IN_SECONDS = 1;

	private long startTimeInMillis;
	private boolean hadRecords = false;
	private KafkaConsumer<K, V> consumer;
	private Handler<ConsumerRecord<K, V>> handler;
	private Handler<ConsumerRecords<K, V>> batchHandler;
	private boolean infinite = false;
	private LagConsumer lagConsumer;
	private RateLimiter lagCheckLimiter;

	public KafkaConsumerClientImpl(Properties props) {
		consumer = new KafkaConsumer<>(props);
	}

	@Override
	public KafkaConsumerClientImpl<K, V> handler(Handler<ConsumerRecord<K, V>> handler) {
		this.handler = handler;
		return this;
	}

	@Override
	public KafkaConsumerClient<K, V> lagConsumer(LagConsumer lc) {
		this.lagCheckLimiter = RateLimiter.create(0.1);
		this.lagConsumer = lc;
		return this;
	}

	@Override
	public KafkaConsumerClientImpl<K, V> batchHandler(Handler<ConsumerRecords<K, V>> batchHandler) {
		this.batchHandler = batchHandler;
		return this;
	}

	@Override
	public KafkaConsumerClient<K, V> infinite(boolean infinite) {
		this.infinite = infinite;
		return this;
	}

	@Override
	public Future<Void> subscribe(List<String> topics) {
		consumer.subscribe(topics);
		startTimeInMillis = System.currentTimeMillis();
		return consume();
	}

	private Future<Void> consume() {
		Promise<Void> emptyConsumptionPromise = Promise.promise();
		consume(emptyConsumptionPromise);
		return emptyConsumptionPromise.future();
	}

	private void consume(Promise<Void> emptyConsumptionPromise) {
		Thread.ofPlatform().name("kafka-consumer").start(() -> {
			while (true) {
				final ConsumerRecords<K, V> records = consumer.poll(ofSeconds(POLL_DURATION_IN_SECONDS));
				boolean hasRecord = handle(records);
				if (!infinite && isEmptyConsumption(hasRecord, emptyConsumptionPromise)) {
					emptyConsumptionPromise.complete();
					// break;
				} else if (hasRecord) {
					hadRecords = true;
				}
				if (lagConsumer != null && lagCheckLimiter.tryAcquire()) {
					lagConsumer.checkLag(consumer);
				}
			}
		});
	}

	private boolean isEmptyConsumption(boolean hasRecord, Promise<Void> emptyConsumptionPromise) {
		return (!hasRecord && !emptyConsumptionPromise.future().isComplete()
				&& (hadRecords || upTimeInSeconds() > MIN_UPTIME_BEFORE_EMPTY_CONSUMPTION));
	}

	private boolean handle(final ConsumerRecords<K, V> records) {
		if (records == null || records.count() == 0) {
			return false;
		}
		logger.debug("consuming {} records", records.count());
		if (Objects.isNull(this.batchHandler) && !Objects.isNull(this.handler)) {
			records.forEach(handler::handle);
		} else if (!Objects.isNull(this.batchHandler)) {
			this.batchHandler.handle(records);
		}
		return true;
	}

	private long upTimeInSeconds() {
		return (System.currentTimeMillis() - startTimeInMillis) / 1000;
	}

}
