/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.common.backup;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.user.api.User;

public class RepositoryBackupPath {
	private static Logger logger = LoggerFactory.getLogger(RepositoryBackupPath.class);

	public static final String JSON_FILE_EXTENSION = ".json";
	public static final String HISTORY_JSON_FILE_EXTENSION = "_history.json";
	public static final String VCF_FILE_EXTENSION = ".vcf";
	public static final String ICS_FILE_EXTENSION = ".ics";
	public static final String ZIP_FILE_EXTENSION = ".zip";
	public static final String JSON_INDEX_FILENAME = "index.json";

	private final Path repository;

	public RepositoryBackupPath(Path repository) {
		this.repository = repository;
	}

	public Path rootPath() {
		return repository;
	}

	public static String getFilePathWithDirectParent(Path filepath) {
		return filepath.getParent().getFileName().resolve(filepath.getFileName()).toString();
	}

	public Path preparePathToWrite(ContainerDescriptor containerDescriptor, String extension) throws IOException {
		Objects.requireNonNull(containerDescriptor);
		Path p = getStoragePath(containerDescriptor, extension);
		Files.createDirectories(p.getParent());
		return p;
	}

	public Path prepareJsonPathToWrite(ContainerDescriptor containerDescriptor) throws IOException {
		Objects.requireNonNull(containerDescriptor);
		Path p = getStoragePath(containerDescriptor, JSON_FILE_EXTENSION);
		Files.createDirectories(p.getParent());
		return p;
	}

	public Path getStoragePath(ContainerDescriptor containerDescriptor, String extension) {
		Objects.requireNonNull(containerDescriptor);
		Objects.requireNonNull(containerDescriptor.uid);
		Objects.requireNonNull(containerDescriptor.owner);
		Objects.requireNonNull(containerDescriptor.domainUid);
		Path p;
		if (containerDescriptor.owner.equals(containerDescriptor.uid)) {
			// Domain calendar
			p = resolveDomainPath(containerDescriptor.domainUid);
		} else {
			// user/resource calendar
			p = resolveUserPath(containerDescriptor.owner);
		}
		return p.resolve(containerDescriptor.uid + extension);
	}

	public Path getStoragePath(String dirEntryUid) {
		Objects.requireNonNull(dirEntryUid);
		return rootPath().resolve(dirEntryUid + JSON_FILE_EXTENSION);
	}

	public Path resolveUserPathForContainer(String ownerUid, String containerUid, String extension) {
		return resolveUserPath(ownerUid).resolve(containerUid + extension);
	}

	public Path resolveDomainPathForContainer(String domainUid, String containerUid, String extension) {
		return resolveDomainPath(domainUid).resolve(containerUid + extension);
	}

	public Path resolveUserPath(String ownerUid) {
		return resolvePath("user").resolve(ownerUid);
	}

	public Path resolveRootUserPath() {
		return resolvePath("user");
	}

	public Path resolveDomainPath(String domainUid) {
		return resolvePath("domain").resolve(domainUid);
	}

	private Path resolvePath(String userOrDomain) {
		return rootPath().resolve(userOrDomain);
	}

	public boolean exists() {
		return Files.exists(rootPath());
	}

	public List<String> getRestorableUserContainerUids(String domainUid, String ownerUid) throws IOException {
		List<String> containerUids = Collections.emptyList();
		try {
			containerUids = readZipFiles(domainUid, ownerUid);
		} catch (Exception e) {
			logger.warn(e.getMessage());
		}
		if (containerUids.isEmpty()) {
			containerUids = readJsonFiles(ownerUid);
		}

		return containerUids;
	}

	private List<String> readZipFiles(String domainUid, String ownerUid) throws Exception {
		Optional<Path> zipFilePathForOwner = getZipFilePathForOwner(domainUid, ownerUid);
		if (zipFilePathForOwner.isEmpty()) {
			return Collections.emptyList();
		}

		List<BackupContainerItemDescriptor<User>> containersFromZip = extractJsonContainerFromZip(ownerUid,
				zipFilePathForOwner.get());

		if (!containersFromZip.isEmpty()) {
			return containersFromZip.stream().map(containerItem -> containerItem.getContainer().uid).toList();
		}

		return Collections.emptyList();
	}

	private List<String> readJsonFiles(String ownerUid) throws IOException {
		Path restorablesPath = resolveUserPath(ownerUid);
		List<BackupContainerItemDescriptor<User>> containerss = (List<BackupContainerItemDescriptor<User>>) JsonReadWrite
				.readJsonFile(JsonReadWrite.readerUser(), restorablesPath);
		return containerss.stream().map(ab -> ab.getContainer().uid).toList();
	}

	private static List<BackupContainerItemDescriptor<User>> extractJsonContainerFromZip(String folderName,
			Path zipFilePath) {
		List<BackupContainerItemDescriptor<User>> userContainers = new ArrayList<>();
		try {
			JsonReadWrite.findJsonFilesWithPattern(folderName, zipFilePath).forEach(byteArray -> {
				try {
					userContainers.add((BackupContainerItemDescriptor<User>) JsonReadWrite
							.readByteArr(JsonReadWrite.readerUser(), byteArray));
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			});
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return userContainers;
	}

	public Optional<Path> getZipFilePathForDirEntry(String domainUid, String ownerUid) throws Exception {
		return Optional.ofNullable(getZipFilePath(domainUid, ownerUid, rootPath()));
	}

	public Optional<Path> getZipFilePathForOwner(String domainUid, String ownerUid) throws Exception {
		return Optional.ofNullable(getZipFilePath(domainUid, ownerUid, resolveRootUserPath()));
	}

	private Path getZipFilePath(String domainUid, String ownerUid, Path zipStoragePath) throws Exception {
		if (!Files.isDirectory(zipStoragePath)) {
			return null;
		}

		Path indexPath = zipStoragePath.resolve(JSON_INDEX_FILENAME);
		if (!Files.exists(indexPath)) {
			return null;
		}

		MailboxIndexJsonList indexJson = readIndexFile(indexPath);
		if (indexJson == null) {
			throw new FileNotFoundException(JSON_INDEX_FILENAME + " File " + indexPath + " not found");
		}

		Optional<String> byOwnerUid = indexJson.mailboxesIndexInfo().stream()
				.filter(i -> i.mailboxUid().equals(ownerUid) && i.domainUid().equals(domainUid))
				.map(i -> i.zipFileName()).findFirst();

		if (byOwnerUid.isEmpty()) {
			throw new FileNotFoundException("Owner " + ownerUid + " not found in " + indexPath);
		}
		return zipStoragePath.resolve(byOwnerUid.get());
	}

	public static Path writeIndexFileToDir(Path workingFolder, List<MailboxIndexJson> mailboxRetList)
			throws IOException {
		Path jsonIndexPath = workingFolder.resolve(JSON_INDEX_FILENAME);
		JsonReadWrite.write(JsonUtils.writer(MailboxIndexJsonList.class), jsonIndexPath,
				new MailboxIndexJsonList(mailboxRetList));
		return jsonIndexPath;
	}

	public static MailboxIndexJsonList readIndexFile(Path indexPath) throws IOException {
		return (MailboxIndexJsonList) JsonReadWrite.readJsonFileObj(JsonUtils.reader(MailboxIndexJsonList.class),
				indexPath);
	}

}
