/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.authentication.service.internal;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientOptions;
import net.bluemind.keycloak.utils.sessions.OpenIdSessionRefresher;
import net.bluemind.keydb.sessiondata.SessionData;
import net.bluemind.keydb.sessiondata.SessionDataStore;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;

public class OpenIdRefreshHandler extends AbstractVerticle {
	private static final Logger logger = LoggerFactory.getLogger(OpenIdRefreshHandler.class);

	public static class OpenIdRefreshHandlerVerticleFactory implements IVerticleFactory, IUniqueVerticleFactory {
		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new OpenIdRefreshHandler();
		}
	}

	private HttpClient httpClient;

	private OpenIdRefreshHandler() {
	}

	@Override
	public void start() throws Exception {
		httpClient = vertx.createHttpClient(new HttpClientOptions().setTrustAll(true).setVerifyHost(false));

		vertx.setPeriodic(60_000, 60_000, tid -> SessionDataStore.get().requeueRefreshSessionId());
		vertx.setPeriodic(90_000, 60_000, tid -> refresh());
	}

	public void refresh() {
		String sessionId;
		while ((sessionId = SessionDataStore.get().getSessionIdToRefresh()) != null) {
			SessionData sessionData = SessionDataStore.get().getIfPresent(sessionId);

			if (sessionData == null) {
				if (logger.isDebugEnabled()) {
					logger.debug("Remove unknown session {} from refresh queue - session logged out ?", sessionId);
				}
				SessionDataStore.get().removeRefreshSessionId(sessionId);
				continue;
			}

			if (sessionData.openIdRefreshStamp > System.currentTimeMillis()) {
				// No refresh needed
				return;
			}

			new OpenIdSessionRefresher(sessionData.realm, sessionData.authKey, sessionData.openIdClientSecret,
					sessionData.jwtToken,
					refreshedJwtToken -> SessionDataStore.get()
							.updateSessionData(sessionData.setOpenId(refreshedJwtToken, sessionData.realm,
									sessionData.openIdClientSecret, sessionData.internalAuth,
									System.currentTimeMillis() + SessionDataStore.SESSIONID_REFRESH_PERIOD)))
					.refresh(httpClient);
		}
	}
}
