package net.bluemind.directory.hollow.datamodel.consumer.multicore;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.util.concurrent.DefaultThreadFactory;
import net.bluemind.directory.hollow.datamodel.AddressBookRecord;
import net.bluemind.directory.hollow.datamodel.OfflineAddressBook;
import net.bluemind.directory.hollow.datamodel.consumer.AddressBookMatcher;
import net.bluemind.directory.hollow.datamodel.consumer.IDirectoryDeserializer;
import net.bluemind.directory.hollow.datamodel.consumer.Query;
import net.bluemind.directory.hollow.datamodel.consumer.Query.QueryType;
import net.bluemind.directory.hollow.datamodel.consumer.SearchResults;
import net.bluemind.directory.hollow.datamodel.consumer.multicore.index.IDeserializerIndex;
import net.bluemind.directory.hollow.datamodel.consumer.multicore.index.IDeserializerIndexMono;
import net.bluemind.directory.hollow.datamodel.consumer.multicore.index.IDeserializerIndexMulti;
import net.bluemind.directory.hollow.datamodel.consumer.multicore.index.IndexCatalog;

public class MultiCoreDirectoryDeserializer implements IDirectoryDeserializer {

	private static final Logger logger = LoggerFactory.getLogger(MultiCoreDirectoryDeserializer.class);

	private static final Set<String> complexQueryKeys = new HashSet<>(Arrays.asList("office", "emails"));
	private static final Map<String, Object> lockByDomains = new ConcurrentHashMap<>();

	private static final ExecutorService executorService = Executors
			.newSingleThreadExecutor(new DefaultThreadFactory("directory-deserializer-loop"));

	private static AddressBookRecordFieldAccessor fieldAccessor = new AddressBookRecordFieldAccessor();
	private Runnable updateVersionListener = this::buildIndex;
	private IMultiCoreDirectoryDeserializerStore store;
	private long version = 0;

	private OfflineAddressBook root = null;

	private IndexCatalog indexCatalog = new IndexCatalog();

	private String domain;

	private static Object getLockByDomain(String domain) {
		return lockByDomains.computeIfAbsent(domain, d -> new Object());
	}

	public MultiCoreDirectoryDeserializer(String domain, IMultiCoreDirectoryDeserializerStore store,
			boolean watchChanges) {
		this.domain = domain;
		this.store = store;

		if (watchChanges) {
			store.watchVersionUpdate(updateVersionListener);
		}
		buildIndexSynchrone();
	}

	public MultiCoreDirectoryDeserializer(String domain, IMultiCoreDirectoryDeserializerStore store) {
		this(domain, store, true);
	}

	public void buildIndex() {
		// we use a thread because of
		// https://github.com/redis/lettuce/wiki/Frequently-Asked-Questions#im-seeing-rediscommandtimeoutexception
		executorService.submit(this::buildIndexSynchrone);
	}

	public void buildIndexSynchrone() {
		synchronized (getLockByDomain(domain)) {
			long updateToVersion = store.getVersion();
			if (updateToVersion > version) {
				logger.warn("Start deserialization for version {} for {}", updateToVersion, this);

				IndexCatalog indexesUpdated = new IndexCatalog();
				store.forEachIndex(updateToVersion, addressBook -> {
					String uid = addressBook.getUid();
					for (IDeserializerIndex index : indexesUpdated.getAll()) {
						try {
							index.add(addressBook, uid);
						} catch (Exception e) {
							logger.error("Deserialization error on {} to {}", uid, index, e);
						}
					}
				});

				root = store.getOfflineAdressBookRecords(updateToVersion);

				indexCatalog = indexesUpdated;
				version = updateToVersion;

				logger.warn("Deserialization done for version {}, {} index created", version,
						indexCatalog.getAll().size());
			}
		}
	}

	@Override
	public Optional<OfflineAddressBook> root() {
		return Optional.ofNullable(root);
	}

	@Override
	public Optional<AddressBookRecord> byDistinguishedName(String distinguishedName) {
		return indexCatalog.getByDistinguishedName()
				.getUid(distinguishedName != null ? distinguishedName.toLowerCase() : null)
				.map(this::mapUidToAddressbookRecord);
	}

	@Override
	public Optional<AddressBookRecord> byUid(String uid) {
		return indexCatalog.getByUid().getUid(uid).map(this::mapUidToAddressbookRecord);
	}

	@Override
	public Optional<AddressBookRecord> byMinimalId(long minimalId) {
		return indexCatalog.getByMinimalId().getUid(String.valueOf(minimalId)).map(this::mapUidToAddressbookRecord);
	}

	@Override
	public Optional<AddressBookRecord> byEmail(String email) {
		return indexCatalog.getByEmail().getUids(email).stream().findFirst().map(this::mapUidToAddressbookRecord);
	}

	@Override
	public Collection<AddressBookRecord> byNameOrEmailPrefix(String value) {
		return store.batchGetAdressBookRecords(version,
				indexCatalog.getByNameOrEmailPrefix().getUids(value != null ? value.toLowerCase() : null));
	}

	@Override
	public Collection<AddressBookRecord> all() {
		return store.getAllAdressBookRecords(version, addressBook -> true);
	}

	private AddressBookRecord mapUidToAddressbookRecord(String uid) {
		return store.getAdressBookRecord(version, uid);
	}

	@Override
	public SearchResults byKind(List<String> kinds, int offset, int limit, Predicate<AddressBookRecord> filter) {
		List<AddressBookRecord> allFilterByKind = kinds.stream() //
				.flatMap(kind -> byKind(kind).stream()) //
				.filter(filter) //
				.sorted((a, b) -> Objects.compare(a.name, b.name, String::compareTo)) //
				.toList();

		int total = allFilterByKind.size();
		if (offset < 0) {
			offset = 0;
		}
		if (limit < 0) {
			limit = total;
		}
		offset = Math.min(total, offset);
		int to = Math.min(total, offset + limit);
		return new SearchResults(total, allFilterByKind.subList(offset, to));
	}

	@Override
	public Collection<AddressBookRecord> search(Query query) {
		if (query.type == QueryType.VALUE && !complexQueryKeys.contains(query.key)) {
			return simpleQuery(query.key, query.value);
		} else {
			return complexQuery(query);
		}
	}

	private Collection<AddressBookRecord> complexQuery(Query query) {
		return this.store.getAllAdressBookRecords(version, filter(query));
	}

	private Predicate<AddressBookRecord> filter(Query query) {
		return (AddressBookRecord entry) -> eval(query, entry);
	}

	private boolean eval(Query query, AddressBookRecord entry) {
		switch (query.type) {
		case VALUE:
			return evalValue(query.key, query.value, entry);
		case AND:
			boolean match = true;
			for (Query child : query.children) {
				match = match && eval(child, entry);
			}
			return match;
		case OR:
			match = false;
			for (Query child : query.children) {
				match = match || eval(child, entry);
			}
			return match;
		default:
			return false;
		}
	}

	private boolean evalValue(String key, String value, AddressBookRecord entry) {
		return AddressBookMatcher.matches(key, value, root(), entry);
	}

	private Collection<String> getUids(String value, IDeserializerIndex index) {
		if (index instanceof IDeserializerIndexMono deserializerIndex) {
			return deserializerIndex.getUid(value).stream().toList();
		} else if (index instanceof IDeserializerIndexMulti deserializerIndexMulti) {
			return deserializerIndexMulti.getUids(value);
		}
		// Not supposed to occurs
		throw new IllegalArgumentException(index.getClass().getName() + " instance not supported");
	}

	@SuppressWarnings("unchecked")
	private List<AddressBookRecord> simpleQuery(String key, String value) {
		Optional<Collection<String>> maybeUids = indexCatalog.getForSimpleQuery(key)
				.map(index -> this.getUids(value, index));
		return maybeUids //
				.map(uids -> store.batchGetAdressBookRecords(version, uids)) //
				.orElseGet(() -> {
					logger.warn("No index found for {}, maybe we should add a new one ?", key);
					return this.store.getAllAdressBookRecords(version,
							addressBook -> Objects.equals(fieldAccessor.getAsString(addressBook, key), value));
				});
	}

	private Collection<AddressBookRecord> byKind(String kind) {
		return store.batchGetAdressBookRecords(version, indexCatalog.getByKind().getUids(kind));
	}

	@Override
	public void stopWatcher() {
		logger.info("Stop watching for new serialization version, current version is {}", version);
		store.stopWatchVersionUpdate();
	}

	@Override
	public boolean isWatcherListening() {
		return store.isWatchingVersionUpdate();
	}

	public long getVersion() {
		return version;
	}

	@Override
	public void forEach(Consumer<AddressBookRecord> cons) {
		store.forEachIndex(version, idxOnly -> {
			AddressBookRecord full = store.getAdressBookRecord(version, idxOnly.uid);
			if (full != null) {
				cons.accept(full);
			}
		});
	}
}
