package net.bluemind.cli.adm;

import java.util.AbstractMap;
import java.util.List;
import java.util.Optional;

import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.config.InstallationId;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.domain.api.IDomains;
import net.bluemind.server.api.IServer;
import net.bluemind.server.api.TagDescriptor;
import picocli.CommandLine.Command;
import picocli.CommandLine.IExitCodeGenerator;

@Command(name = "orphan-body-repair", description = "delete orphan mail bodies")
public class OrphanBodyCommand implements ICmdLet, Runnable, IExitCodeGenerator {

	public static class Reg implements ICmdLetRegistration {
		@Override
		public Optional<String> group() {
			return Optional.of("maintenance");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return OrphanBodyCommand.class;
		}
	}

	private CliContext ctx;
	protected CliUtils cliUtils;
	private int exitCode = 0;

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}

	@Override
	public void run() {
		ctx.info("Deleting orphan bodies...");
		IServiceProvider provider = ctx.adminApi();
		IServer serverService = provider.instance(IServer.class, InstallationId.getIdentifier());
		IDomains domainService = provider.instance(IDomains.class);
		List<AbstractMap.SimpleEntry<String, String>> serversByDomains = domainService.all().stream().map(d -> d.uid)
				.filter(uid -> !"global.virt".equals(uid)).flatMap(domainUid -> {
					List<String> servers = serverService.byAssignment(domainUid, TagDescriptor.mail_imap.getTag());
					return servers.stream()
							.map(serverUid -> new AbstractMap.SimpleEntry<String, String>(domainUid, serverUid));
				}).toList();

		for (AbstractMap.SimpleEntry<String, String> serverByDomain : serversByDomains) {
			String domainUid = serverByDomain.getKey();
			String serverUid = serverByDomain.getValue();

			ctx.info("Deleting orphan bodies on {} / {}", domainUid, serverUid);
			CyrusPartition cyrusPartition = CyrusPartition.forServerAndDomain(serverUid, domainUid);
			IDbMessageBodies bodiesService = provider.instance(IDbMessageBodies.class, cyrusPartition.name);
			bodiesService.deleteOrphans();
		}
	}

	@Override
	public int getExitCode() {
		return exitCode;
	}

}
