/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.user.impl;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.ContainerStore;
import net.bluemind.core.jdbc.JdbcActivator;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.api.TaskStatus;
import net.bluemind.core.task.service.TaskUtils;
import net.bluemind.dataprotect.addressbook.RestoreUserAddressbooks;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.RestoreDefinition;
import net.bluemind.dataprotect.calendar.RestoreUserCalendars;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.directory.RestoreDirWorker;
import net.bluemind.dataprotect.mailbox.AbstractRestoreActionProvider;
import net.bluemind.dataprotect.notes.RestoreUserNotes;
import net.bluemind.dataprotect.todolist.RestoreUserTodolists;
import net.bluemind.dataprotect.webappdata.RestoreUserWebAppData;
import net.bluemind.group.api.Group;
import net.bluemind.group.api.IGroup;
import net.bluemind.group.api.Member;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.MailFilter;
import net.bluemind.system.api.Credential;
import net.bluemind.system.api.IInternalCredentials;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.TagChanges;
import net.bluemind.user.api.ChangePassword;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.IUserSettings;
import net.bluemind.user.api.User;
import net.bluemind.user.persistence.UserStore;

public class UserRestoreWorker extends RestoreDirWorker<User> {
	private static final Logger logger = LoggerFactory.getLogger(UserRestoreWorker.class);

	public UserRestoreWorker(DataProtectGeneration backup, IMonitoredRestoreRestorableItem restorableItem) {
		super(backup, restorableItem);
	}

	public void restoreUserPassword(BmContext backupContext, long oldUserId, long newUserId) throws SQLException {
		DataSource ds = backupContext.getDataSource();
		ContainerStore cs = new ContainerStore(backupContext, ds, backupContext.getSecurityContext());
		Container domain = cs.get(restorableItem.domain());
		UserStore userStore = new UserStore(ds, domain);
		String pwd = userStore.getPassword(oldUserId);

		ds = JdbcActivator.getInstance().getDataSource();
		cs = new ContainerStore(backupContext, ds, backupContext.getSecurityContext());
		domain = cs.get(restorableItem.domain());
		userStore = new UserStore(JdbcActivator.getInstance().getDataSource(), domain);
		Item i = Item.create(restorableItem.entryUid(), null);
		i.id = newUserId;
		userStore.setPassword(i, pwd, true);
	}

	public void restoreUserPassword(String pwd) {
		if (pwd == null) {
			return;
		}
		live.instance(IUser.class, restorableItem.domain()).setPassword(restorableItem.liveEntryUid(),
				ChangePassword.create(pwd));
	}

	@Override
	public void restoreEntryDependantContainers() {
		logger.info("Restoring user mbox {}({}):{}", restorableItem.entryUid(), restorableItem.liveEntryUid(),
				restorableItem.displayName());

		RestoreDefinition restoreBox = new RestoreDefinition(AbstractRestoreActionProvider.RESTORE_REPLACE_IDENTIFIER,
				backup.id, restorableItem.item());
		executeTask(restoreBox);

		logger.info("Restoring user addressbooks {}:{}", restorableItem.entryUid(), restorableItem.displayName());

		RestoreDefinition restoreBooks = new RestoreDefinition(RestoreUserAddressbooks.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreBooks);

		logger.info("Restoring user calendars {}:{}", restorableItem.entryUid(), restorableItem.displayName());

		RestoreDefinition restoreCalendars = new RestoreDefinition(RestoreUserCalendars.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreCalendars);

		logger.info("Restoring user todolists {}:{}", restorableItem.entryUid(), restorableItem.displayName());

		RestoreDefinition restoreTodolists = new RestoreDefinition(RestoreUserTodolists.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreTodolists);

		logger.info("Restoring user notes {}:{}", restorableItem.entryUid(), restorableItem.displayName());

		RestoreDefinition restoreNotes = new RestoreDefinition(RestoreUserNotes.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreNotes);

		RestoreDefinition restoreWebAppData = new RestoreDefinition(RestoreUserWebAppData.RESTORE_IDENTIFIER, backup.id,
				restorableItem.item());
		executeTask(restoreWebAppData);
	}

	@Override
	public ItemValue<User> createOrUpdateLiveEntry(ItemValue<User> backupUser) throws InterruptedException {
		IUser userServiceLive = live.instance(IUser.class, restorableItem.domain());

		ItemValue<User> liveUser = userServiceLive.getComplete(restorableItem.entryUid());
		if (liveUser != null) {
			userServiceLive.update(liveUser.uid, backupUser.value);
			restorableItem.setLiveEntryUid(liveUser.uid);
			return liveUser;
		}

		liveUser = userServiceLive.byLogin(backupUser.value.login);
		if (liveUser != null) {
			TaskRef deleteTask = userServiceLive.delete(liveUser.uid);
			TaskStatus taskDeleteStatus = TaskUtils.waitForInterruptible(live, deleteTask);
			if (taskDeleteStatus.state.succeed) {
				restorableItem.monitorLog("Delete user '{}' succeed {}", backupUser.value.login,
						taskDeleteStatus.result);
			}
		}

		userServiceLive.create(restorableItem.entryUid(), backupUser.value);
		liveUser = userServiceLive.getComplete(restorableItem.entryUid());
		restorableItem.setLiveEntryUid(liveUser.uid);
		return liveUser;
	}

	public void restoreUserFilters(MailFilter mailFilter) {
		live.instance(IMailboxes.class, restorableItem.domain()).setMailboxFilter(restorableItem.liveEntryUid(),
				mailFilter);
	}

	public void restoreUserTags(List<TagChanges.ItemAdd> userTags) {
		live.instance(ITags.class, ITagUids.TYPE + "_" + restorableItem.liveEntryUid())
				.updates(TagChanges.create(userTags, Collections.emptyList(), Collections.emptyList()));
	}

	public void restoreUserCredentials(List<Credential> credentials) {
		live.instance(IInternalCredentials.class, restorableItem.domain()).addUserCredentials(restorableItem.liveEntryUid(),
				credentials);
	}

	public void restoreGroupMembership(List<ItemValue<Group>> memberOf) {
		IGroup groupService = live.instance(IGroup.class, restorableItem.domain());
		for (ItemValue<Group> group : memberOf) {
			// add to group, if group still exists
			ItemValue<Group> groupItem = groupService.getComplete(group.uid);
			if (groupItem != null
					&& !groupService.getMembers(group.uid).contains(Member.user(restorableItem.liveEntryUid()))) {
				groupService.add(group.uid, Arrays.asList(Member.user(restorableItem.liveEntryUid())));
			}
		}
	}

	public void restoreUserSettings(Map<String, String> settings) {
		live.instance(IUserSettings.class, restorableItem.domain()).set(restorableItem.liveEntryUid(), settings);
	}

}
