/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.system.service.helper;

import java.math.BigInteger;
import java.util.Calendar;
import java.util.TimeZone;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

public class Totp {
	public static final int INTERVAL_SECONDS = 30;
	private static final int[] DELAY_WINDOW = { 0, -1, 1 };
	public static final String ALGORITHM = "HmacSHA1";
	public static final int NUMBER_DIGITS = 6;
	// 0 1 2 3 4 5 6 7 8
	private static final int[] DIGITS_POWER = { 1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000 };

	public boolean validateTOTP(String token, String secret) {
		if (token == null || token.isEmpty() || secret == null || secret.isEmpty()) {
			return false;
		}

		long currentInterval = getCurrentInterval();

		for (int i : DELAY_WINDOW) {
			long adjustedInterval = currentInterval + i;

			String candidate = generateOTP(secret.getBytes(), step2Bytes(adjustedInterval));
			if (candidate.equals(token)) {
				return true;
			}
		}

		return false;
	}

	private byte[] step2Bytes(long step) {
		StringBuilder steps = new StringBuilder(Long.toHexString(step).toUpperCase());
		// Just get a 16 digit string
		while (steps.length() < 16) {
			steps.insert(0, "0");
		}

		return hexStr2Bytes(steps.toString());
	}

	/**
	 * This method converts HEX string to Byte[]
	 *
	 * @param hex the HEX string
	 * @return A byte array
	 */
	private byte[] hexStr2Bytes(String hex) {
		// Adding one byte to get the right conversion
		// values starting with "0" can be converted
		byte[] bArray = new BigInteger("10" + hex, 16).toByteArray();

		// Copy all the REAL bytes, not the "first"
		byte[] ret = new byte[bArray.length - 1];
		System.arraycopy(bArray, 1, ret, 0, ret.length);
		return ret;
	}

	private long getCurrentInterval() {
		return (Calendar.getInstance(TimeZone.getTimeZone("UTC")).getTimeInMillis() / 1000) / INTERVAL_SECONDS;
	}

	public String generateCurrentOTP(String key) {
		return generateOTP(key.getBytes(), step2Bytes(getCurrentInterval()));
	}

	/**
	 * This method generates an OTP value for the given set of parameters.
	 *
	 * @param key  the shared secret, HEX encoded
	 * @param step a value that reflects a time
	 * @return A numeric String in base 10 that includes return digits
	 * @throws java.security.GeneralSecurityException
	 *
	 */
	private String generateOTP(byte[] key, byte[] step) {
		byte[] hash = hmacSha1(key, step);

		// put selected bytes into result int
		int offset = hash[hash.length - 1] & 0xf;

		int binary = ((hash[offset] & 0x7f) << 24) | ((hash[offset + 1] & 0xff) << 16)
				| ((hash[offset + 2] & 0xff) << 8) | (hash[offset + 3] & 0xff);

		int otp = binary % DIGITS_POWER[NUMBER_DIGITS];

		StringBuilder result = new StringBuilder(Integer.toString(otp));
		while (result.length() < NUMBER_DIGITS) {
			result.insert(0, "0");
		}

		return result.toString();
	}

	/**
	 * This method uses the JCE to provide the crypto algorithm. HMAC computes a
	 * Hashed Message Authentication Code with the crypto hash algorithm as a
	 * parameter.
	 *
	 * @param crypto   the crypto algorithm (HmacSHA1, HmacSHA256, HmacSHA512)
	 * @param keyBytes the bytes to use for the HMAC key
	 * @param text     the message or text to be authenticated.
	 * @throws java.security.NoSuchAlgorithmException
	 *
	 * @throws java.security.InvalidKeyException
	 *
	 */
	private byte[] hmacSha1(byte[] keyBytes, byte[] text) {
		byte[] value;

		try {
			Mac hmac = Mac.getInstance(ALGORITHM);
			SecretKeySpec macKey = new SecretKeySpec(keyBytes, "RAW");

			hmac.init(macKey);

			value = hmac.doFinal(text);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		return value;
	}
}
