/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.exchange.mapi.category;

import java.io.ByteArrayInputStream;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.NodeList;

import io.netty.buffer.ByteBufUtil;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.tag.api.Tag;
import net.bluemind.tag.api.TagChanges;
import net.bluemind.tag.api.TagChanges.ItemAdd;
import net.bluemind.tag.api.TagChanges.ItemDelete;
import net.bluemind.tag.api.TagChanges.ItemModify;
import net.bluemind.utils.DOMUtils;

public class CategoryList {

	private static final Logger logger = LoggerFactory.getLogger(CategoryList.class);
	private final Set<Category> categories;
	private final JsonObject json;
	private final static Map<Integer, RGB> defaultColorMap;
	private TagChanges changes;

	private CategoryList(JsonObject json, Set<Category> categories) {
		this.json = json;
		this.categories = categories;
	}

	static {
		Map<Integer, RGB> map = new HashMap<>(24);
		map.put(-1, new RGB((byte) 255, (byte) 255, (byte) 255, "No color"));
		map.put(0, new RGB((byte) 214, (byte) 37, (byte) 46, "Red"));
		map.put(1, new RGB((byte) 240, (byte) 100, (byte) 21, "Orange"));
		map.put(2, new RGB((byte) 255, (byte) 202, (byte) 76, "Peach"));
		map.put(3, new RGB((byte) 255, (byte) 254, (byte) 61, "Yellow"));
		map.put(4, new RGB((byte) 74, (byte) 182, (byte) 63, "Green"));
		map.put(5, new RGB((byte) 64, (byte) 189, (byte) 149, "Teal"));

		map.put(6, new RGB((byte) 133, (byte) 154, (byte) 82, "Olive"));
		map.put(7, new RGB((byte) 50, (byte) 103, (byte) 184, "Blue"));
		map.put(8, new RGB((byte) 97, (byte) 61, (byte) 180, "Purple"));
		map.put(9, new RGB((byte) 163, (byte) 78, (byte) 120, "Maroon"));
		map.put(10, new RGB((byte) 196, (byte) 204, (byte) 221, "Steel"));
		map.put(11, new RGB((byte) 140, (byte) 156, (byte) 189, "Dark Steel"));
		map.put(12, new RGB((byte) 196, (byte) 196, (byte) 196, "Gray"));
		map.put(13, new RGB((byte) 165, (byte) 165, (byte) 165, "Dark Gray"));
		map.put(14, new RGB((byte) 28, (byte) 28, (byte) 28, "Black"));
		map.put(15, new RGB((byte) 175, (byte) 30, (byte) 37, "Dark Red"));
		map.put(16, new RGB((byte) 177, (byte) 79, (byte) 13, "Dark orange"));
		map.put(17, new RGB((byte) 171, (byte) 123, (byte) 5, "Dark peach"));
		map.put(18, new RGB((byte) 153, (byte) 148, (byte) 0, "Dark yellow"));
		map.put(19, new RGB((byte) 53, (byte) 121, (byte) 43, "Dark green"));
		map.put(20, new RGB((byte) 46, (byte) 125, (byte) 100, "Dark teal"));
		map.put(21, new RGB((byte) 95, (byte) 108, (byte) 58, "Dark olive"));
		map.put(22, new RGB((byte) 42, (byte) 81, (byte) 145, "Dark blue"));
		map.put(23, new RGB((byte) 80, (byte) 50, (byte) 143, "Dark purple"));
		map.put(24, new RGB((byte) 130, (byte) 55, (byte) 95, "Dark maroon"));
		defaultColorMap = map;
	}

	public static CategoryList fromJson(JsonObject json) throws Exception {
		JsonObject properties = json.getJsonObject("setProperties");
		String hexXml = properties.getString("PidTagRoamingXmlStream");

		byte[] decodeHexDump = ByteBufUtil.decodeHexDump(hexXml);
		Document document = DOMUtils.parse(new ByteArrayInputStream(decodeHexDump));
		NodeList xmlCategories = document.getElementsByTagName("category");
		Set<Category> categories = new HashSet<>(xmlCategories.getLength());

		for (int i = 0; i < xmlCategories.getLength(); i++) {
			NamedNodeMap attributes = xmlCategories.item(i).getAttributes();
			categories.add(new Category(attributes.getNamedItem("name").getNodeValue(),
					Integer.parseInt(attributes.getNamedItem("color").getNodeValue()),
					attributes.getNamedItem("guid").getNodeValue()));
		}

		return new CategoryList(json, categories);
	}

	public String getEncodedFai() {
		JsonObject properties = json.getJsonObject("setProperties");
		try {
			properties.put("PidTagRoamingXmlStream", encodeCategories());
		} catch (Exception e) {
			logger.info("Cannot reintegrate categories", e);
		}
		return json.encode();
	}

	public boolean mergeClientChanges(List<ItemValue<Tag>> existingTags) {
		changes = TagChanges.create(new ArrayList<>(), new ArrayList<>(), new ArrayList<>());
		for (Category clientTag : categories) {
			String hex = getColor(clientTag.colorCode()).toHexColor();
			existingTags.stream().filter(tag -> tag.value.label.equals(clientTag.name)).findFirst()
					.ifPresentOrElse(serverTag -> {
						if (!serverTag.value.color.equalsIgnoreCase(hex)) {
							// don't update fallback color back to server
							if (!(isUnsupportedColor(serverTag.value.color) && isFallbackColor(clientTag.colorCode))) {
								changes.modify.add(ItemModify.create(serverTag.uid, Tag.create(clientTag.name(), hex)));
							}
						}
					}, () -> {
						changes.add
								.add(ItemAdd.create(UUID.randomUUID().toString(), Tag.create(clientTag.name(), hex)));
					});
		}
		changes.delete.addAll(existingTags.stream().filter(
				existing -> categories.stream().noneMatch(clientTag -> clientTag.name.equals(existing.value.label)))
				.map(item -> ItemDelete.create(item.uid)).toList());

		return !changes.add.isEmpty() || !changes.modify.isEmpty() || !changes.delete.isEmpty();
	}

	public void mergeServerChanges(ItemValue<Tag> tag) {
		Optional<Category> existing = categories.stream().filter(cat -> cat.name().equals(tag.value.label)).findFirst();
		existing.ifPresentOrElse((cat) -> {
			categories.remove(cat);
			categories.add(new Category(cat.name(), getColorCode(tag.value.color), cat.guid()));
		}, () -> {
			categories.add(new Category(tag.value.label, getColorCode(tag.value.color), UUID.randomUUID().toString()));
		});
	}

	public void removeDeletedTags(List<ItemValue<Tag>> serverTags) {
		categories.removeIf(cat -> serverTags.stream().noneMatch(sTag -> sTag.value.label.equals(cat.name())));
	}

	private String encodeCategories() throws Exception {

		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document doc = builder.newDocument();

		// Create root element
		Element root = doc.createElement("categories");
		if (!categories.isEmpty()) {
			root.setAttribute("default", categories.iterator().next().name());
		}
		root.setAttribute("lastSavedSession", "0");
		root.setAttribute("lastSavedTime", "2025-01-01T00:00:00.000");
		root.setAttribute("xmlns", "CategoryList.xsd");
		doc.appendChild(root);

		for (Category cat : categories) {
			Element category = doc.createElement("category");
			category.setAttribute("name", cat.name());
			category.setAttribute("color", "" + cat.colorCode());
			category.setAttribute("keyboardShortcut", "0");
			category.setAttribute("usageCount", "1");
			category.setAttribute("lastTimeUsedNotes", "1601-01-01T00:00:00.000");
			category.setAttribute("lastTimeUsedJournal", "1601-01-01T00:00:00.000");
			category.setAttribute("lastTimeUsedContacts", "1601-01-01T00:00:00.000");
			category.setAttribute("lastTimeUsedTasks", "1601-01-01T00:00:00.000");
			category.setAttribute("lastTimeUsedCalendar", "1601-01-01T00:00:00.000");
			category.setAttribute("lastTimeUsedMail", "1601-01-01T00:00:00.000");
			category.setAttribute("lastTimeUsed", "1601-01-01T00:00:00.000");
			category.setAttribute("lastSessionUsed", "0");
			category.setAttribute("guid", cat.guid());
			root.appendChild(category);
		}

		TransformerFactory transformerFactory = TransformerFactory.newInstance();
		Transformer transformer = transformerFactory.newTransformer();
		transformer.setOutputProperty(OutputKeys.INDENT, "yes");
		transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "1");
		transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");

		StringWriter writer = new StringWriter();
		DOMSource source = new DOMSource(doc);
		StreamResult result = new StreamResult(writer);
		transformer.transform(source, result);

		return ByteBufUtil.hexDump(writer.toString().getBytes());
	}

	private boolean isFallbackColor(int colorCode) {
		return colorCode == getFallbackColor();
	}

	private int getColorCode(String color) {
		Optional<Entry<Integer, RGB>> entry = defaultColorMap.entrySet().stream()
				.filter(rgb -> rgb.getValue().toHexColor().equals(color)).findFirst();
		if (entry.isPresent()) {
			return entry.get().getKey();
		} else {
			return getFallbackColor();
		}
	}

	private boolean isUnsupportedColor(String color) {
		return defaultColorMap.values().stream().noneMatch(rgb -> rgb.toHexColor().equals(color));
	}

	private int getFallbackColor() {
		return 12;
	}

	public TagChanges getChanges() {
		if (changes == null) {
			throw new IllegalStateException("tag changes have not been calculated yet");
		}
		return changes;
	}

	public Set<Category> getCategories() {
		return categories;
	}

	public RGB getColor(int colorCode) {
		return defaultColorMap.getOrDefault(colorCode, defaultColorMap.get(getFallbackColor()));
	}

	public static record Category(String name, int colorCode, String guid) {

		public String guid() {
			return guid.startsWith("{") ? guid : "{" + guid + "}";
		}

	}

	public static record RGB(byte r, byte g, byte b, String name) {

		public String toHexColor() {
			return ByteBufUtil.hexDump(new byte[] { r, g, b });
		}

	}

}
