/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.exchange.mapi.service.internal;

import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.eventbus.Message;
import io.vertx.core.json.JsonObject;
import net.bluemind.calendar.api.ICalendar;
import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.exchange.mapi.api.IMapiFolderAssociatedInformation;
import net.bluemind.exchange.mapi.api.IMapiMailbox;
import net.bluemind.exchange.mapi.api.MapiFAI;
import net.bluemind.exchange.mapi.api.MapiReplica;
import net.bluemind.exchange.mapi.category.CategoryList;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.Tag;

public class MapiCategoryListSync extends AbstractVerticle {

	Logger logger = LoggerFactory.getLogger(MapiCategoryListSync.class);

	@Override
	public void start() throws Exception {
		super.vertx.eventBus().consumer("tags.changed", (Message<JsonObject> msg) -> {
			vertx.executeBlocking(() -> {
				tagChanged(msg.body().getString("containerUid"));
				return null;
			});
		});
	}

	public void tagChanged(String tagContainerUid) {
		logger.debug("tags of container {} changed", tagContainerUid);
		ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		ContainerDescriptor container = provider.instance(IContainers.class).get(tagContainerUid);
		if (isDomainTagsContainer(container)) {
			return;
		}
		IMapiMailbox mapiApi = provider.instance(IMapiMailbox.class, container.domainUid, container.owner);
		MapiReplica mapiReplica = mapiApi.get();
		if (mapiReplica != null) {
			String replica = mapiReplica.localReplicaGuid;
			IMapiFolderAssociatedInformation faiApi = provider.instance(IMapiFolderAssociatedInformation.class,
					replica);
			List<ItemValue<MapiFAI>> fais = faiApi.getByFolderId(ICalendarUids.defaultUserCalendar(container.owner));
			if (!fais.isEmpty()) { // we dont try to sync if the user does not use outlook
				logger.debug("owner {} has {} calendar FAIs", fais.size(), container.owner);
				Optional<ItemValue<MapiFAI>> faiOptional = fais.stream()
						.filter(f -> f.value.faiJson.contains("IPM.Configuration.CategoryList")).findAny();
				faiOptional.ifPresent(fai -> {
					String json = fai.value.faiJson;
					try {
						CategoryList categoryList = CategoryList.fromJson(new JsonObject(json));
						ITags tags = provider.instance(ITags.class, tagContainerUid);
						List<ItemValue<Tag>> serverTags = tags.all();
						serverTags.forEach(tag -> categoryList.mergeServerChanges(tag));
						categoryList.removeDeletedTags(serverTags);
						fai.value.faiJson = categoryList.getEncodedFai();
						faiApi.store(fai.internalId, fai.value);
						touchUserCalendar(provider, container);
					} catch (Exception e) {
						logger.warn("Cannot reintegrate tag changes into MAPI FAI", e);
					}
				});
			}
		}
	}

	private boolean isDomainTagsContainer(ContainerDescriptor container) {
		return container.owner.equals(container.domainUid);
	}

	private void touchUserCalendar(ServerSideServiceProvider provider, ContainerDescriptor container) {
		ICalendar service = provider.instance(ICalendar.class, ICalendarUids.defaultUserCalendar(container.owner));
		List<String> all = service.all();
		if (!all.isEmpty()) {
			service.touch(all.getLast());
		}
	}

}
