/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.serdes.settings;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import net.bluemind.eas.dto.IPreviousRequestsKnowledge;
import net.bluemind.eas.dto.OptionalParams;
import net.bluemind.eas.dto.settings.OofMessage;
import net.bluemind.eas.dto.settings.OofMessage.Audience;
import net.bluemind.eas.dto.settings.OofState;
import net.bluemind.eas.dto.settings.SettingsRequest;
import net.bluemind.eas.dto.settings.SettingsRequest.DeviceInformation;
import net.bluemind.eas.dto.settings.SettingsRequest.DeviceInformation.Set;
import net.bluemind.eas.dto.settings.SettingsRequest.Oof;
import net.bluemind.eas.dto.settings.SettingsRequest.Oof.Get;
import net.bluemind.eas.dto.settings.SettingsRequest.UserInformation;
import net.bluemind.eas.dto.settings.SettingsResponse.Oof.BodyType;
import net.bluemind.eas.serdes.IEasRequestParser;
import net.bluemind.eas.utils.EasLogUser;

public class SettingsRequestParser implements IEasRequestParser<SettingsRequest> {

	private static final Logger logger = LoggerFactory.getLogger(SettingsRequestParser.class);

	@Override
	public SettingsRequest parse(OptionalParams optParams, Document doc, IPreviousRequestsKnowledge past, String user) {
		SettingsRequest sr = new SettingsRequest();
		Element settings = doc.getDocumentElement();
		NodeList children = settings.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node n = children.item(i);
			if (n.getNodeType() != Node.ELEMENT_NODE) {
				continue;
			}
			Element child = (Element) n;
			String childName = child.getNodeName();
			switch (childName) {
			case "Oof": // set / get
				sr.oof = parseOof(child, user);
				break;
			case "DeviceInformation": // set
				sr.deviceInformation = parseDeviceInformation(child, user);
				break;
			case "UserInformation": // get
				sr.userInformation = parseUserInformation(child, user);
				break;
			default:
				EasLogUser.logWarnAsUser(user, logger, "Not managed Settings child {}", child);
				break;
			}
		}
		return sr;
	}

	private UserInformation parseUserInformation(Element oofElem, String user) {
		UserInformation ui = new UserInformation();
		NodeList children = oofElem.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node n = children.item(i);
			if (n.getNodeType() != Node.ELEMENT_NODE) {
				continue;
			}
			Element child = (Element) n;
			String childName = child.getNodeName();
			switch (childName) {
			case "Get":
				ui.get = new SettingsRequest.UserInformation.Get();
				break;
			default:
				EasLogUser.logWarnAsUser(user, logger, "Not managed Oof child: '{}'", child);
				break;
			}
		}
		return ui;
	}

	private Oof parseOof(Element oofElem, String user) {
		Oof oof = new Oof();
		NodeList children = oofElem.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node n = children.item(i);
			if (n.getNodeType() != Node.ELEMENT_NODE) {
				continue;
			}
			Element child = (Element) n;

			String childName = child.getNodeName();
			switch (childName) {
			case "Get":
				oof.get = parseOofGet(child, user);
				break;
			case "Set":
				oof.set = parseOofSet(child, user);
				break;
			default:
				EasLogUser.logWarnAsUser(user, logger, "Not managed Oof child: '{}'", child);
				break;
			}
		}
		return oof;
	}

	public DeviceInformation parseDeviceInformation(Element diElem, String user) {
		DeviceInformation di = new DeviceInformation();
		NodeList children = diElem.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node n = children.item(i);
			if (n.getNodeType() != Node.ELEMENT_NODE) {
				continue;
			}
			Element child = (Element) n;

			String childName = child.getNodeName();
			switch (childName) {
			case "Set":
				di.set = parseDevInfSet(child, user);
				break;
			default:
				EasLogUser.logWarnAsUser(user, logger, "Not managed DeviceInformation child: '{}'", child);
				break;
			}
		}
		return di;
	}

	/**
	 * <pre>
	 *     <Set>
	 *       <Model>iPad3C1</Model>
	 *       <OS>iOS 9.0.1 13A404</OS>
	 *       <OSLanguage>fr-FR</OSLanguage>
	 *       <FriendlyName>Noir iPad</FriendlyName>
	 *     </Set>
	 * </pre>
	 * 
	 * @param devInfElem
	 * @return
	 */
	private Set parseDevInfSet(Element devInfElem, String user) {
		Set set = new SettingsRequest.DeviceInformation.Set();
		NodeList children = devInfElem.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node n = children.item(i);
			if (n.getNodeType() != Node.ELEMENT_NODE) {
				continue;
			}
			Element child = (Element) n;

			String childName = child.getNodeName();
			switch (childName) {
			case "Model":
				set.model = child.getTextContent();
				break;
			case "OS":
				set.os = child.getTextContent();
				break;
			case "IMEI":
				set.imei = child.getTextContent();
				break;
			case "MobileOperator":
				set.mobileOperator = child.getTextContent();
				break;
			case "UserAgent":
				set.userAgent = child.getTextContent();
				break;
			case "PhoneNumber":
				set.phoneNumber = child.getTextContent();
				break;
			case "OSLanguage":
				set.osLanguage = child.getTextContent();
				break;
			case "FriendlyName":
				set.friendlyName = child.getTextContent();
				break;
			default:
				EasLogUser.logWarnAsUser(user, logger, "Not managed Oof.Get child: '{}'", child);
				break;
			}
		}
		return set;
	}

	private Get parseOofGet(Element oofGetElem, String user) {
		Get get = new SettingsRequest.Oof.Get();
		NodeList children = oofGetElem.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node n = children.item(i);
			if (n.getNodeType() != Node.ELEMENT_NODE) {
				continue;
			}
			Element child = (Element) n;
			String childName = child.getNodeName();
			switch (childName) {
			case "BodyType":
				get.bodyType = child.getTextContent();
				break;
			default:
				EasLogUser.logWarnAsUser(user, logger, "Not managed Oof.Get child: '{}'", child);
				break;
			}
		}
		return get;
	}

	private net.bluemind.eas.dto.settings.SettingsRequest.Oof.Set parseOofSet(Element oofSetElem, String user) {
		net.bluemind.eas.dto.settings.SettingsRequest.Oof.Set set = new SettingsRequest.Oof.Set();
		NodeList children = oofSetElem.getChildNodes();
		for (int i = 0; i < children.getLength(); i++) {
			Node n = children.item(i);
			if (n.getNodeType() != Node.ELEMENT_NODE) {
				continue;
			}
			Element child = (Element) n;
			String childName = child.getNodeName();
			if (childName.equals("OofState")) {
				String val = child.getTextContent();
				set.oofState = OofState.fromXml(val);
			}
			if (childName.equals("StartTime")) {
				String val = child.getTextContent();
				set.startTime = val;
			}
			if (childName.equals("EndTime")) {
				String val = child.getTextContent();
				set.endTime = val;
			}
			if (childName.equals("OofMessage")) {
				OofMessage message = parseOofMessage(child);
				if (set.oofMessages.stream().noneMatch(m -> m.audience.equals(message.audience))) {
					set.oofMessages.add(message);
				}
			}

		}
		return set;
	}

	private OofMessage parseOofMessage(Element element) {
		int enabled = 0;
		String replyMessage = "";
		BodyType bodyType = BodyType.TEXT;
		Audience audience = Audience.AppliesToExternalUnknown;

		NodeList oofMessageNodes = element.getChildNodes();
		for (int i = 0; i < oofMessageNodes.getLength(); i++) {
			Node node = oofMessageNodes.item(i);
			String value = node.getTextContent();
			switch (node.getNodeName()) {
			case "Enabled" -> enabled = Integer.parseInt(value);
			case "ReplyMessage" -> replyMessage = value;
			case "BodyType" -> bodyType = BodyType.valueOf(value);
			case "AppliesToInternal", "AppliesToExternalKnown", "AppliesToExternalUnknown" ->
				audience = Audience.valueOf(node.getNodeName());
			}
		}
		return new OofMessage(enabled, replyMessage, bodyType, audience);

	}

}
