/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cli.mail;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.hash.Hashing;

import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.IMailboxFoldersByContainer;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.replica.api.AppendTx;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.cli.utils.CliUtils.ResolvedMailbox;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.delivery.conversationreference.api.IConversationReference;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mailbox.api.Mailbox.Type;
import net.bluemind.utils.ProgressPrinter;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

@Command(name = "import-emls", description = "Import eml files")
public class ImportEmlsCommand implements ICmdLet, Runnable {

	private CliContext ctx;
	private CliUtils cliUtils;

	@Option(names = "--email", required = true, description = "Email address of the mailbox to restore from")
	public String email;

	@Option(names = "--resolve-cyrus-path", required = false, defaultValue = "FALSE", description = "Try to detect root from a cyrus mail server hierarchy")
	public boolean resolveCyrusPath = false;

	@Parameters(paramLabel = "<path>", description = "Root backup folder")
	public String rootFolder;

	private String filenamePattern = ".eml";

	@Override
	public void run() {

		ResolvedMailbox mailboxByEmail = cliUtils.getMailboxByEmail(email);
		if (mailboxByEmail == null) {
			ctx.error("Mailbox {} not found", email);
			System.exit(1);
		}

		Services services = initializeServices(mailboxByEmail);

		List<String> rootFolders = null;
		if (resolveCyrusPath) {
			rootFolders = resolveCyrusRootPath(rootFolder, mailboxByEmail);
			filenamePattern = ".";
		} else {
			rootFolders = Arrays.asList(rootFolder);
		}

		String restoreFolderName = "restored-" + DateTimeFormatter.ISO_INSTANT.format(Instant.now());

		for (String root : rootFolders) {
			ctx.info("Handling root folder {}", root);
			List<Path> folders = new ArrayList<>(getFolderHierarchy(root));

			sortFoldersByHierarchyDepths(folders);
			for (Path folder : folders) {
				ctx.info("Handling  folder {}", folder.toFile().getAbsolutePath());
				Path rootPath = PathResolver.getResolver(mailboxByEmail).resolveRoot(root, folder);
				MailboxFolder newFolder = new MailboxFolder();
				boolean isInbox = isInbox(rootPath, folder);
				newFolder.name = isInbox ? restoreFolderName : folder.getFileName().toString();
				String fullName = isInbox ? restoreFolderName
						: restoreFolderName + "/" + rootPath.relativize(folder).toString();
				newFolder.fullName = PathResolver.getResolver(mailboxByEmail).getFolderFullName(mailboxByEmail.mailbox,
						fullName);
				try {
					services.foldersService.createBasic(newFolder);
				} catch (ServerFault e) {
					// IGNORE
				}
				ItemValue<MailboxFolder> resolvedFolder = services.foldersService.byName(newFolder.fullName);
				IDbMailboxRecords recordsApi = ctx.adminApi().instance(IDbMailboxRecords.class, resolvedFolder.uid);

				restoreFolder(services, recordsApi, resolvedFolder, folder);
			}
		}
	}

	private List<Path> getFolderHierarchy(String root) {
		try (Stream<Path> stream = Files.walk(Paths.get(new File(root).toURI()))) {
			return stream.filter(Files::isDirectory).toList();
		} catch (IOException e) {
			ctx.error("Cannot traverse folder: {}", e.getMessage());
			return Collections.emptyList();
		}
	}

	private void restoreFolder(Services services, IDbMailboxRecords recordsApi, ItemValue<MailboxFolder> resolvedFolder,
			Path folder) {
		List<Mail> emls = Arrays.asList(folder.toFile().listFiles(this::isMail)).stream().map(eml -> {
			String bodyGuid;
			try {
				bodyGuid = Hashing.sha1().hashBytes(Files.readAllBytes(eml.toPath())).toString();
			} catch (IOException e) {
				ctx.error("Cannot load body file {}:{}", eml, ctx.toStack(e));
				return null;
			}
			return new Mail(eml, new Date(eml.lastModified()), bodyGuid);
		}).filter(obj -> !Objects.isNull(obj)).toList();
		ProgressPrinter progress = new ProgressPrinter(emls.size(), 1_000, 5);
		List<List<Mail>> partitioned = Lists.partition(emls, 32);
		partitioned.stream().forEach(msgList -> {
			for (Mail mail : msgList) {
				importMail(services, recordsApi, resolvedFolder, mail);
				progress.add();
				if (progress.shouldPrint()) {
					ctx.info("Mail import progress: {}", progress);
				}
			}
		});
	}

	private void importMail(Services services, IDbMailboxRecords recordsApi, ItemValue<MailboxFolder> resolvedFolder,
			Mail mail) {
		try (InputStream instream = Files.newInputStream(mail.file.toPath())) {
			ItemVersion added = createMail(services, recordsApi, resolvedFolder, mail);

			if (added.id <= 0) {
				ctx.error("Unable to inject message {}", mail.file.getAbsolutePath());
			}
		} catch (Exception e) {
			ctx.error("error while handling: {}:{}", mail.file.getAbsolutePath(), ctx.toStack(e));
		}
	}

	private ItemVersion createMail(Services services, IDbMailboxRecords recordsApi,
			ItemValue<MailboxFolder> resolvedFolder, Mail mail) throws IOException {
		AppendTx appendTx = services.mailboxApi.prepareAppend(resolvedFolder.internalId, 1);
		Date bodyDeliveryDate = mail.date == null ? new Date(appendTx.internalStamp) : mail.date;
		net.bluemind.core.api.Stream stream = GenericStream.simpleValue(Files.readAllBytes(mail.file.toPath()), b -> b);
		services.bodiesApi.createWithDeliveryDate(mail.guid, bodyDeliveryDate.getTime(), stream);
		MessageBody messageBody = services.bodiesApi.getComplete(mail.guid);
		Set<String> references = (messageBody.references != null) ? Sets.newHashSet(messageBody.references)
				: Sets.newHashSet();
		Long conversationId = null;
		if (services.conversationReferenceApi != null) {
			conversationId = services.conversationReferenceApi.lookup(messageBody.messageId, references);
		}
		MailboxRecord rec = new MailboxRecord();
		rec.imapUid = appendTx.imapUid;
		rec.internalDate = bodyDeliveryDate;
		rec.messageBody = mail.guid;
		rec.conversationId = conversationId;
		rec.flags = List.of(MailboxItemFlag.System.Seen.value());
		rec.lastUpdated = rec.internalDate;
		return recordsApi.create(appendTx.imapUid + ".", rec);
	}

	private List<String> resolveCyrusRootPath(String rootFolder, ResolvedMailbox mbox) {
		try (Stream<Path> stream = Files.walk(Paths.get(new File(rootFolder).toURI()))) {
			Optional<Path> domainFolder = stream
					.filter(f -> Files.isDirectory(f) && f.getFileName().toString().equals(mbox.domainUid)).findAny();
			if (domainFolder.isEmpty()) {
				ctx.error("Cannot identify domain folder in spool, starting from {}", rootFolder);
				System.exit(1);
			}
			return PathResolver.getResolver(mbox).resolve(domainFolder.get(), mbox);
		} catch (Exception e) {
			ctx.error("Cannot traverse folder: {}", e.getMessage());
			System.exit(1);
			return Collections.emptyList();
		}
	}

	private Services initializeServices(ResolvedMailbox mailboxByEmail) {
		IMailboxFolders folderService = ctx.adminApi().instance(IMailboxFoldersByContainer.class, IMailReplicaUids
				.subtreeUid(mailboxByEmail.domainUid, mailboxByEmail.mailbox.value.type, mailboxByEmail.mailbox.uid));
		CyrusPartition cyrusPartition = CyrusPartition.forServerAndDomain(mailboxByEmail.mailbox.value.dataLocation,
				mailboxByEmail.domainUid);
		IDbReplicatedMailboxes mailboxApi = ctx.adminApi().instance(IDbByContainerReplicatedMailboxes.class,
				IMailReplicaUids.subtreeUid(mailboxByEmail.domainUid, mailboxByEmail.mailbox));
		IDbMessageBodies bodiesApi = ctx.adminApi().instance(IDbMessageBodies.class, cyrusPartition.name);
		IConversationReference conversationReferenceApi = null;
		if (mailboxByEmail.mailbox.value.type == Type.user) {
			conversationReferenceApi = ctx.adminApi().instance(IConversationReference.class, mailboxByEmail.domainUid,
					mailboxByEmail.mailbox.uid);
		}
		return new Services(mailboxApi, bodiesApi, conversationReferenceApi, folderService);

	}

	private boolean isInbox(Path root, Path folder) {
		return root.toFile().getAbsolutePath().equals(folder.toFile().getAbsolutePath());
	}

	private void sortFoldersByHierarchyDepths(List<Path> folders) {
		folders.sort((a, b) -> a.toFile().getAbsolutePath().compareTo(b.toFile().getAbsolutePath()));
	}

	private boolean isMail(File file) {
		return file.isFile() && file.getName().endsWith(filenamePattern);
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("mail");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return ImportEmlsCommand.class;
		}
	}

	static record Mail(File file, Date date, String guid) {

	}

	static record Services(IDbReplicatedMailboxes mailboxApi, IDbMessageBodies bodiesApi,
			IConversationReference conversationReferenceApi, IMailboxFolders foldersService) {
	}

	interface IResolvePath {
		List<String> resolve(Path domainRoot, ResolvedMailbox mbox);

		String getFolderFullName(ItemValue<Mailbox> mbox, String fullName);

		Path resolveRoot(String baseDir, Path folder);
	}

	static class PathResolver {

		private PathResolver() {
		}

		static IResolvePath getResolver(ResolvedMailbox mbox) {
			switch (mbox.mailbox.value.type) {
			case mailshare:
				return new MailsharePathResolver();
			case user:
			default:
				return new UserPathResolver();
			}
		}

		static class PathResolution {

			protected char mapLetter(String mbox) {
				char mboxLetter = mbox.charAt(0);
				if (!Character.isLetter(mboxLetter)) {
					return 'q';
				}

				return Character.toLowerCase(mboxLetter);
			}

		}

		static class MailsharePathResolver extends PathResolution implements IResolvePath {

			@Override
			public List<String> resolve(Path domainRoot, ResolvedMailbox mbox) {
				File[] prefixFolders = domainRoot.toFile().listFiles(f -> f.isDirectory() && f.getName().length() == 1);
				List<String> mboxFolders = new ArrayList<>();
				for (File prefixFolder : prefixFolders) {
					File[] boxfolders = prefixFolder //
							.listFiles(f -> f.isDirectory() && f.getName().equals(mbox.mailbox.value.name));
					mboxFolders.addAll(Arrays.asList(boxfolders).stream().map(File::getAbsolutePath).toList());
				}
				return mboxFolders;
			}

			@Override
			public Path resolveRoot(String baseDir, Path folder) {
				File file = folder.toFile();
				if (baseDir.equals(file.getAbsolutePath()) && file.listFiles(File::isDirectory).length == 0) {
					return folder;
				} else {
					return Paths.get(new File(baseDir).toURI());
				}
			}

			@Override
			public String getFolderFullName(ItemValue<Mailbox> mbox, String fullName) {
				return mbox.value.name + "/" + fullName;
			}

		}

		static class UserPathResolver extends PathResolution implements IResolvePath {

			@Override
			public List<String> resolve(Path domainRoot, ResolvedMailbox mbox) {
				char mboxLetter = mapLetter(mbox.mailbox.value.name);
				String boxName = mbox.mailbox.value.name.replace('.', '^');
				StringBuilder path = new StringBuilder(domainRoot.toString());
				path.append("/");
				path.append(mboxLetter);
				path.append("/user/");
				path.append(boxName);
				return Arrays.asList(path.toString());
			}

			@Override
			public Path resolveRoot(String baseDir, Path folder) {
				return Paths.get(baseDir).toAbsolutePath();
			}

			@Override
			public String getFolderFullName(ItemValue<Mailbox> mbox, String fullName) {
				return fullName;
			}

		}

	}

}
