/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.cli.mail;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.hash.Hashing;

import net.bluemind.backend.mail.api.IMailFinder;
import net.bluemind.backend.mail.api.MailFinderQuery;
import net.bluemind.backend.mail.api.SearchResult;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.core.container.api.IContainerManagement;
import picocli.CommandLine;
import picocli.CommandLine.ArgGroup;
import picocli.CommandLine.Command;
import picocli.CommandLine.Model.CommandSpec;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;
import picocli.CommandLine.Spec;

@Command(name = "delete-message", description = "Delete specific message using messageId/subject/from")
public class DeleteMessageCommand implements ICmdLet, Runnable {

	private CliContext ctx;
	private CliUtils cliUtils;

	@Spec
	CommandSpec spec;

	@Parameters(paramLabel = "<domain>", description = "the domain")
	public String domainUid;

	@Option(names = "--message-id", required = false, description = "messageId of the message to delete")
	public String messageId;

	@Option(names = "--from", required = false, description = "the sender of the message to delete")
	public String from;

	@Option(names = "--subject", required = false, description = "the subject of the message to delete")
	public String subject;

	@Option(names = "--deleteIf", required = false, description = "use the previously given hash to delete the messages")
	public String deleteIfHash;

	@ArgGroup(exclusive = true)
	DateOption dateOption;

	private static class DateOption {
		@Option(names = "--before", required = true, description = "search for mails before this date - yyyy-MM-dd")
		String before;

		@Option(names = "--after", required = true, description = "search for mails after this date - yyyy-MM-dd")
		String after;
	}

	@Override
	public void run() {
		String domain = cliUtils.getDomainUidByDomain(domainUid);

		IMailFinder mailFinderApi = ctx.adminApi().instance(IMailFinder.class, domain);

		MailFinderQuery query = new MailFinderQuery();
		query.from = from;
		query.messageId = messageId;
		query.subject = subject;

		if (dateOption != null && (messageId == null && from == null && subject == null)) {
			ctx.error("Cannot have only --before or --after option. Must have also --messageId or --subject or --from");
		}

		if (dateOption != null && dateOption.before != null) {
			checkDate(dateOption.before);
			query.before = dateOption.before;
		}
		if (dateOption != null && dateOption.after != null) {
			checkDate(dateOption.after);
			query.after = dateOption.after;
		}

		SearchResult searchResult = mailFinderApi.search(query);

		if (searchResult.totalResults == 0) {
			ctx.info("No results found");
			return;
		}

		String reducedItemId = searchResult.results.stream().map(r -> r.itemId).sorted().map(i -> String.valueOf(i))
				.reduce((str1, str2) -> str1 + str2).get();
		String hash = Hashing.murmur3_128().hashBytes(reducedItemId.getBytes()).toString();

		if (deleteIfHash == null) {
			Map<String, String> ownerMailMap = new HashMap<>();
			ctx.info("Messages to remove:");
			searchResult.results.forEach(r -> {
				ownerMailMap.computeIfAbsent(r.containerUid,
						k -> ctx.adminApi().instance(IContainerManagement.class, k).getDescriptor().ownerDisplayname);
				ctx.info("for owner '" + ownerMailMap.get(r.containerUid) + "' - subject:'" + r.subject
						+ "' - sender: '" + r.from.address + "'");
			});
			CommandLine commandLine = spec.commandLine();
			List<String> originalArgs = commandLine.getParseResult().originalArgs();
			ctx.warn("\r\nIf you realy want to delete the items listed above, use the following cli command:");
			ctx.warn("bm-cli " + originalArgs.stream().reduce((str1, str2) -> str1 + " " + str2).get() + " --deleteIf "
					+ hash);
			return;
		}

		if (!deleteIfHash.equals(hash)) {
			ctx.warn("Parameter for deleteIf '" + deleteIfHash + "' do not match with computed hash: '" + hash + "'");
			return;
		}

		Map<String, List<Long>> groupingMap = searchResult.results.stream().collect(
				Collectors.groupingBy(s -> s.containerUid, Collectors.mapping(s -> s.itemId, Collectors.toList())));
		groupingMap.forEach((containerUid, value) -> {

			IDbMailboxRecords recordsApi = ctx.adminApi().instance(IDbMailboxRecords.class,
					IMailReplicaUids.uniqueId(containerUid));
			value.forEach(v -> {
				recordsApi.deleteById(v);
				ctx.error("Message '" + v + "' deleted");
			});
		});
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("mail");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return DeleteMessageCommand.class;
		}
	}

	private void checkDate(String date) {
		try {
			(new SimpleDateFormat("yyyy-MM-dd")).parse(date);
		} catch (ParseException e) {
			throw new CliException("Invalid date format: " + date);
		}
	}
}
