/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.ui.common.client.forms.acl;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;
import java.util.stream.Collectors;

import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.directory.api.DirEntry;

public class AclEntity {

	private final DirEntry entry;
	private final Verb verb;
	private final Verb delVerb;

	public AclEntity(DirEntry entry, List<AccessControlEntry> verbs) {
		this.entry = entry;

		if (verbs == null) {
			this.verb = null;
			this.delVerb = null;
			return;
		}
		
		Map<Boolean, Optional<Verb>> partitioned = verbs.stream()
			.map(ace -> ace.verb)
			.collect(Collectors.partitioningBy(
				AclEntity::isHandledVerb,
				Collectors.collectingAndThen(
					Collectors.toList(),
					list -> list.isEmpty() ? Optional.empty() : Optional.of(list.get(0))
				)
			));
		
		this.verb = partitioned.get(true).orElse(null);
		this.delVerb = partitioned.get(false).orElse(null);
	}

	private static boolean isHandledVerb(Verb verb) {
		return verb == Verb.Freebusy || verb == Verb.Invitation || verb == Verb.Read || verb == Verb.Write 
		|| verb == Verb.Manage || verb == Verb.All || verb == Verb.Visible;
	}

	public AclEntity(DirEntry entry, Verb verb, Verb delVerb) {
		this.entry = entry;
		this.verb = verb;
		this.delVerb = delVerb;
	}

	public DirEntry getEntry() {
		return entry;
	}

	public Verb getVerb() {
		return verb;
	}

	public Verb getDelVerb() {
		return delVerb;
	}

	@Override
	public String toString() {
		return "AclEntity : " + entry.entryUid + " (" + entry.displayName + ") "
			+ "Verb => " + verb
			+ "Delegation => " + delVerb;
	}
}
