/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.state.internals;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OffsetCheckpoint {
    private static final Logger LOG = LoggerFactory.getLogger(OffsetCheckpoint.class);
    private static final Pattern WHITESPACE_MINIMUM_ONCE = Pattern.compile("\\s+");
    private static final int VERSION = 0;
    public static final long OFFSET_UNKNOWN = -4L;
    private final File file;
    private final Object lock;

    public OffsetCheckpoint(File file) {
        this.file = file;
        this.lock = new Object();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void write(Map<TopicPartition, Long> offsets) throws IOException {
        if (offsets.isEmpty()) {
            Utils.delete(this.file);
            return;
        }
        Object object = this.lock;
        synchronized (object) {
            File temp = new File(this.file.getAbsolutePath() + ".tmp");
            LOG.trace("Writing tmp checkpoint file {}", (Object)temp.getAbsolutePath());
            FileOutputStream fileOutputStream = new FileOutputStream(temp);
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)fileOutputStream, StandardCharsets.UTF_8));){
                OffsetCheckpoint.writeIntLine(writer, 0);
                OffsetCheckpoint.writeIntLine(writer, offsets.size());
                for (Map.Entry<TopicPartition, Long> entry : offsets.entrySet()) {
                    TopicPartition tp = entry.getKey();
                    Long offset = entry.getValue();
                    if (this.isValid(offset)) {
                        OffsetCheckpoint.writeEntry(writer, tp, offset);
                        continue;
                    }
                    LOG.error("Received offset={} to write to checkpoint file for {}", (Object)offset, (Object)tp);
                    throw new IllegalStateException("Attempted to write a negative offset to the checkpoint file");
                }
                writer.flush();
                fileOutputStream.getFD().sync();
            }
            LOG.trace("Swapping tmp checkpoint file {} {}", (Object)temp.toPath(), (Object)this.file.toPath());
            Utils.atomicMoveWithFallback(temp.toPath(), this.file.toPath());
        }
    }

    static void writeIntLine(BufferedWriter writer, int number) throws IOException {
        writer.write(Integer.toString(number));
        writer.newLine();
    }

    static void writeEntry(BufferedWriter writer, TopicPartition part, long offset) throws IOException {
        writer.write(part.topic());
        writer.write(32);
        writer.write(Integer.toString(part.partition()));
        writer.write(32);
        writer.write(Long.toString(offset));
        writer.newLine();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public Map<TopicPartition, Long> read() throws IOException {
        Object object = this.lock;
        synchronized (object) {
            try (BufferedReader reader = Files.newBufferedReader(this.file.toPath(), StandardCharsets.UTF_8);){
                int version = this.readInt(reader);
                switch (version) {
                    case 0: {
                        int expectedSize = this.readInt(reader);
                        HashMap<TopicPartition, Long> offsets = new HashMap<TopicPartition, Long>();
                        String line = reader.readLine();
                        while (line != null) {
                            String[] pieces = WHITESPACE_MINIMUM_ONCE.split(line);
                            if (pieces.length != 3) {
                                throw new IOException(String.format("Malformed line in offset checkpoint file: '%s'.", line));
                            }
                            String topic = pieces[0];
                            int partition = Integer.parseInt(pieces[1]);
                            TopicPartition tp = new TopicPartition(topic, partition);
                            long offset = Long.parseLong(pieces[2]);
                            if (this.isValid(offset)) {
                                offsets.put(tp, offset);
                            } else {
                                LOG.warn("Read offset={} from checkpoint file for {}", (Object)offset, (Object)tp);
                                --expectedSize;
                            }
                            line = reader.readLine();
                        }
                        if (offsets.size() != expectedSize) {
                            throw new IOException(String.format("Expected %d entries but found only %d", expectedSize, offsets.size()));
                        }
                        HashMap<TopicPartition, Long> hashMap = offsets;
                        return hashMap;
                    }
                }
                throw new IllegalArgumentException("Unknown offset checkpoint version: " + version);
            }
            catch (NoSuchFileException e) {
                return Collections.emptyMap();
            }
        }
    }

    private int readInt(BufferedReader reader) throws IOException {
        String line = reader.readLine();
        if (line == null) {
            throw new EOFException("File ended prematurely.");
        }
        return Integer.parseInt(line);
    }

    public void delete() throws IOException {
        Files.deleteIfExists(this.file.toPath());
    }

    public String toString() {
        return this.file.getAbsolutePath();
    }

    private boolean isValid(long offset) {
        return offset >= 0L || offset == -4L;
    }
}

