/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals.tasks;

import java.time.Duration;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.internals.KafkaFutureImpl;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.errors.TaskMigratedException;
import org.apache.kafka.streams.internals.StreamsConfigUtils;
import org.apache.kafka.streams.processor.internals.ProcessingThread;
import org.apache.kafka.streams.processor.internals.ReadOnlyTask;
import org.apache.kafka.streams.processor.internals.StreamTask;
import org.apache.kafka.streams.processor.internals.Task;
import org.apache.kafka.streams.processor.internals.TaskExecutionMetadata;
import org.apache.kafka.streams.processor.internals.tasks.TaskExecutor;
import org.apache.kafka.streams.processor.internals.tasks.TaskManager;
import org.slf4j.Logger;

public class DefaultTaskExecutor
implements TaskExecutor {
    private final Time time;
    private final String name;
    private final TaskManager taskManager;
    private final TaskExecutionMetadata taskExecutionMetadata;
    private final Logger log;
    private StreamTask currentTask = null;
    private TaskExecutorThread taskExecutorThread = null;

    public DefaultTaskExecutor(TaskManager taskManager, String name, Time time, TaskExecutionMetadata taskExecutionMetadata) {
        this.time = time;
        this.name = name;
        this.taskManager = taskManager;
        this.taskExecutionMetadata = taskExecutionMetadata;
        LogContext logContext = new LogContext(name);
        this.log = logContext.logger(DefaultTaskExecutor.class);
    }

    @Override
    public String name() {
        return this.name;
    }

    @Override
    public void start() {
        if (this.taskExecutorThread == null) {
            this.taskExecutorThread = new TaskExecutorThread(this.name);
            this.taskExecutorThread.start();
        }
    }

    @Override
    public boolean isRunning() {
        return this.taskExecutorThread != null && this.taskExecutorThread.isAlive();
    }

    @Override
    public void requestShutdown() {
        if (this.taskExecutorThread != null) {
            this.taskExecutorThread.shutdownRequested.set(true);
        }
    }

    @Override
    public void awaitShutdown(Duration timeout) {
        if (this.taskExecutorThread != null) {
            try {
                this.taskExecutorThread.join(timeout.toMillis());
                if (this.taskExecutorThread.isAlive()) {
                    throw new StreamsException("State updater thread did not shutdown within the timeout");
                }
                this.taskExecutorThread = null;
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        }
    }

    @Override
    public ReadOnlyTask currentTask() {
        return this.currentTask != null ? new ReadOnlyTask(this.currentTask) : null;
    }

    @Override
    public KafkaFuture<StreamTask> unassign() {
        KafkaFutureImpl<StreamTask> future = new KafkaFutureImpl<StreamTask>();
        if (this.taskExecutorThread != null) {
            this.log.debug("Asking {} to hand back task", (Object)this.taskExecutorThread.getName());
            if (!this.taskExecutorThread.taskReleaseRequested.compareAndSet(null, future)) {
                throw new IllegalStateException("There was already a task release request registered");
            }
            if (!this.taskExecutorThread.isAlive()) {
                this.log.debug("Completing future, because task executor was just shut down");
                future.complete(null);
            } else {
                this.taskManager.signalTaskExecutors();
            }
        } else {
            this.log.debug("Tried to unassign but no thread is running");
            future.complete(null);
        }
        return future;
    }

    private class TaskExecutorThread
    extends Thread
    implements ProcessingThread {
        private final AtomicBoolean shutdownRequested;
        private final AtomicReference<KafkaFutureImpl<StreamTask>> taskReleaseRequested;
        private final Logger log;

        public TaskExecutorThread(String name) {
            super(name);
            this.shutdownRequested = new AtomicBoolean(false);
            this.taskReleaseRequested = new AtomicReference<Object>(null);
            String logPrefix = String.format("%s ", name);
            LogContext logContext = new LogContext(logPrefix);
            this.log = logContext.logger(DefaultTaskExecutor.class);
        }

        @Override
        public void run() {
            this.log.info("Task executor thread started");
            try {
                while (!this.shutdownRequested.get()) {
                    try {
                        this.runOnce(DefaultTaskExecutor.this.time.milliseconds());
                    }
                    catch (StreamsException e) {
                        this.handleException(e);
                    }
                    catch (Exception e) {
                        this.handleException(new StreamsException(e));
                    }
                }
            }
            finally {
                KafkaFutureImpl taskReleaseFuture;
                if (DefaultTaskExecutor.this.currentTask != null) {
                    this.log.debug("Releasing task {} due to shutdown.", (Object)DefaultTaskExecutor.this.currentTask.id());
                    this.unassignCurrentTask();
                }
                if ((taskReleaseFuture = (KafkaFutureImpl)this.taskReleaseRequested.getAndSet(null)) != null) {
                    this.log.debug("Asked to return current task, but shutting down.");
                    taskReleaseFuture.complete(null);
                }
                this.log.info("Task executor thread shutdown");
            }
        }

        private void handleTaskReleaseRequested() {
            KafkaFutureImpl taskReleaseFuture = this.taskReleaseRequested.getAndSet(null);
            if (taskReleaseFuture != null) {
                if (DefaultTaskExecutor.this.currentTask != null) {
                    this.log.debug("Releasing task {} upon request.", (Object)DefaultTaskExecutor.this.currentTask.id());
                    StreamTask unassignedTask = this.unassignCurrentTask();
                    taskReleaseFuture.complete(unassignedTask);
                } else {
                    this.log.debug("Asked to return current task, but returned current task already.");
                    taskReleaseFuture.complete(null);
                }
            }
        }

        private void handleException(StreamsException e) {
            if (DefaultTaskExecutor.this.currentTask == null) {
                throw e;
            }
            DefaultTaskExecutor.this.taskManager.setUncaughtException(e, DefaultTaskExecutor.this.currentTask.id());
            this.log.debug("Releasing task {} due to uncaught exception.", (Object)DefaultTaskExecutor.this.currentTask.id());
            this.unassignCurrentTask();
        }

        private void runOnce(long nowMs) {
            this.handleTaskReleaseRequested();
            if (DefaultTaskExecutor.this.currentTask == null) {
                DefaultTaskExecutor.this.currentTask = DefaultTaskExecutor.this.taskManager.assignNextTask(DefaultTaskExecutor.this);
            }
            if (DefaultTaskExecutor.this.currentTask == null) {
                try {
                    DefaultTaskExecutor.this.taskManager.awaitProcessableTasks(this.shutdownRequested::get);
                }
                catch (InterruptedException interruptedException) {}
            } else {
                boolean progressed = false;
                if (DefaultTaskExecutor.this.taskExecutionMetadata.canProcessTask(DefaultTaskExecutor.this.currentTask, nowMs) && DefaultTaskExecutor.this.currentTask.isProcessable(nowMs) && this.processTask(DefaultTaskExecutor.this.currentTask, nowMs, DefaultTaskExecutor.this.time)) {
                    this.log.trace("processed a record for {}", (Object)DefaultTaskExecutor.this.currentTask.id());
                    progressed = true;
                }
                if (DefaultTaskExecutor.this.taskExecutionMetadata.canPunctuateTask(DefaultTaskExecutor.this.currentTask)) {
                    if (DefaultTaskExecutor.this.currentTask.maybePunctuateStreamTime()) {
                        this.log.trace("punctuated stream time for task {} ", (Object)DefaultTaskExecutor.this.currentTask.id());
                        progressed = true;
                    }
                    if (DefaultTaskExecutor.this.currentTask.maybePunctuateSystemTime()) {
                        this.log.trace("punctuated system time for task {} ", (Object)DefaultTaskExecutor.this.currentTask.id());
                        progressed = true;
                    }
                }
                if (!progressed) {
                    this.log.debug("Releasing task {} because we are not making progress.", (Object)DefaultTaskExecutor.this.currentTask.id());
                    this.unassignCurrentTask();
                }
            }
        }

        private boolean processTask(Task task, long now, Time time) {
            boolean processed = false;
            try {
                processed = task.process(now);
                if (processed) {
                    this.log.trace("Successfully processed task {}", (Object)task.id());
                    task.clearTaskTimeout();
                    if (DefaultTaskExecutor.this.taskExecutionMetadata.hasNamedTopologies() && DefaultTaskExecutor.this.taskExecutionMetadata.processingMode() != StreamsConfigUtils.ProcessingMode.EXACTLY_ONCE_V2) {
                        DefaultTaskExecutor.this.taskExecutionMetadata.addToSuccessfullyProcessed(task);
                    }
                }
            }
            catch (TimeoutException timeoutException) {
                task.maybeInitTaskTimeoutOrThrow(now, timeoutException);
                this.log.error(String.format("Could not complete processing records for %s due to the following exception; will move to next task and retry later", task.id()), (Throwable)timeoutException);
            }
            catch (TaskMigratedException e) {
                this.log.info("Failed to process stream task {} since it got migrated to another thread already. Will trigger a new rebalance and close all tasks as zombies together.", (Object)task.id());
                throw e;
            }
            catch (StreamsException e) {
                this.log.error(String.format("Failed to process stream task %s due to the following error:", task.id()), (Throwable)e);
                e.setTaskId(task.id());
                throw e;
            }
            catch (RuntimeException e) {
                this.log.error(String.format("Failed to process stream task %s due to the following error:", task.id()), (Throwable)e);
                throw new StreamsException(e, task.id());
            }
            finally {
                task.recordProcessBatchTime(time.milliseconds() - now);
            }
            return processed;
        }

        private StreamTask unassignCurrentTask() {
            if (DefaultTaskExecutor.this.currentTask == null) {
                throw new IllegalStateException("Does not own any task while being ask to unassign from task manager");
            }
            if (!DefaultTaskExecutor.this.taskManager.hasUncaughtException(DefaultTaskExecutor.this.currentTask.id())) {
                try {
                    DefaultTaskExecutor.this.currentTask.flush();
                }
                catch (StreamsException e) {
                    this.log.error(String.format("Failed to flush stream task %s due to the following error:", DefaultTaskExecutor.this.currentTask.id()), (Throwable)e);
                    e.setTaskId(DefaultTaskExecutor.this.currentTask.id());
                    DefaultTaskExecutor.this.taskManager.setUncaughtException(e, DefaultTaskExecutor.this.currentTask.id());
                }
                catch (RuntimeException e) {
                    this.log.error(String.format("Failed to flush stream task %s due to the following error:", DefaultTaskExecutor.this.currentTask.id()), (Throwable)e);
                    DefaultTaskExecutor.this.taskManager.setUncaughtException(new StreamsException(e, DefaultTaskExecutor.this.currentTask.id()), DefaultTaskExecutor.this.currentTask.id());
                }
            }
            DefaultTaskExecutor.this.taskManager.unassignTask(DefaultTaskExecutor.this.currentTask, DefaultTaskExecutor.this);
            StreamTask retTask = DefaultTaskExecutor.this.currentTask;
            DefaultTaskExecutor.this.currentTask = null;
            return retTask;
        }
    }
}

