/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.backup;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.collect.Lists;

import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IDbByContainerReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.dataprotect.mailbox.deserializer.CyrusSdsBackupFolder;
import net.bluemind.dataprotect.mailbox.deserializer.CyrusSdsBackupMailbox;
import net.bluemind.dataprotect.mailbox.deserializer.CyrusSdsBackupMessage;
import net.bluemind.dataprotect.sdsspool.SdsDataProtectSpool;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mailshare.api.IMailshare;
import net.bluemind.mailshare.api.Mailshare;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.tag.api.ITagUids;
import net.bluemind.tag.api.ITags;
import net.bluemind.tag.api.Tag;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public abstract class MailboxBackupWorker implements IMailboxBackupWorker {
	private static final int PARTITIONS = 32;

	protected static Logger logger = LoggerFactory.getLogger(MailBackupWorker.class);
	protected MailBackupWorker mailWorker;

	public MailboxBackupWorker(MailBackupWorker mailWorker) {
		this.mailWorker = mailWorker;
	}

	@Override
	public void setBackupStore(SdsDataProtectSpool backupStore) {
		mailWorker.setBackupStore(backupStore);
	}

	@Override
	public void setProductionStores(Map<String, ISdsSyncStore> productionStores) {
		mailWorker.setProductionStores(productionStores);
	}

	private MailboxIndexJsonList runIt(String domainUid, List<String> dirEntriesUids) {
		if (dirEntriesUids.isEmpty()) {
			return new MailboxIndexJsonList(Collections.emptyList());
		}
		List<MailboxIndexJson> mailboxesIndexInfo = new ArrayList<>();
		long timeout = CoreConfig.get().getLong(CoreConfig.DataProtect.BACKUP_SDS_PART_TIME);
		Map<Integer, List<String>> partitionedDirEntriesList = dirEntriesUids.stream()
				.collect(Collectors.groupingBy(uid -> Math.abs(uid.hashCode() % PARTITIONS)));

		try (var executor = Executors.newFixedThreadPool(
				Math.min(partitionedDirEntriesList.size(), Runtime.getRuntime().availableProcessors()))) {

			var futures = partitionedDirEntriesList.entrySet().stream()
					.map(partition -> CompletableFuture
							.runAsync(() -> partTreatment(mailboxesIndexInfo, domainUid, partition), executor))
					.toList();

			CompletableFuture.allOf(futures.toArray(CompletableFuture[]::new)).orTimeout(timeout, TimeUnit.HOURS)
					.join();
		}

		return new MailboxIndexJsonList(mailboxesIndexInfo);
	}

	public record MailboxIndexTotal(List<MailboxIndexJson> mailboxesIndexInfo) {
	}

	public MailboxIndexTotal partitionTreatment(String domainUid, String zipHash, List<String> partition) {
		List<MailboxIndexJson> mailboxToBackup = new ArrayList<>();
		String zipFileName = CommonBackupWorker.zipFilename(getZipFileName(), zipHash);
		partition.forEach(uid -> {
			MailboxIndexJson backupSdsMailbox = backupSdsMailbox(uid);
			if (backupSdsMailbox != null) {
				backupSdsMailbox.zipFileName(zipFileName);
				mailboxToBackup.add(backupSdsMailbox);
			}
		});
		return new MailboxIndexTotal(mailboxToBackup);
	}

	private void partTreatment(List<MailboxIndexJson> mailboxesIndexInfo, String domainUid,
			Entry<Integer, List<String>> partition) {
		String zipHash = String.valueOf(partition.getKey());
		MailboxIndexTotal mailboxIndexAll = partitionTreatment(domainUid, zipHash, partition.getValue());
		if (mailboxIndexAll != null && mailboxIndexAll.mailboxesIndexInfo != null) {
			mailboxesIndexInfo.addAll(mailboxIndexAll.mailboxesIndexInfo);
		}
	}

	@Override
	public MailboxIndexJsonList runIt(ItemValue<Domain> domain, List<String> values, boolean downloadEmailContent) {
		mailWorker.downloadEmailContent = downloadEmailContent;
		mailWorker.domain = domain;
		return runIt(domain.uid, values);
	}

	private MailboxIndexJson backupSdsMailbox(String uid) {
		DirEntry diruser = mailWorker.serviceProvider.instance(IDirectory.class, mailWorker.domain.uid)
				.findByEntryUid(uid);
		return backupSdsMailbox(diruser, mailWorker.domain, mailWorker.downloadEmailContent);
	}

	protected record SdsMailboxRecord(String login, Mailbox mailbox, String mailboxUid, String mailboxName,
			String dataLocation, String filename, Path outputPath) {
	}

	private SdsMailboxRecord createForMailshare(DirEntry de, ItemValue<Domain> domain) {
		ItemValue<Mailshare> mailshare = mailWorker.serviceProvider.instance(IMailshare.class, domain.uid)
				.getComplete(de.entryUid);
		String login = mailshare.value.name;
		String dataLocation = mailshare.value.dataLocation;
		Mailbox mailbox = mailshare.value.toMailbox();
		String mailboxUid = mailshare.uid;
		String mailboxName = mailbox.name;
		String filename = String.format("mailshare_%s@%s.json", mailshare.value.name, domain.value.defaultAlias);
		Path outputPath = Paths.get(mailWorker.workingFolder.toAbsolutePath().toString(),
				String.format("mailshare_%s@%s.json", mailshare.value.name, domain.value.defaultAlias));
		return new SdsMailboxRecord(login, mailbox, mailboxUid, mailboxName, dataLocation, filename, outputPath);
	}

	private SdsMailboxRecord createForMailbox(DirEntry de, ItemValue<Domain> domain) {
		IUser userApi = mailWorker.serviceProvider.instance(IUser.class, domain.uid);
		ItemValue<User> user = userApi.getComplete(de.entryUid);
		String login = user.value.login;
		String dataLocation = user.value.dataLocation;
		ItemValue<Mailbox> mailboxItem = mailWorker.serviceProvider.instance(IMailboxes.class, domain.uid)
				.getComplete(user.uid);
		Mailbox mailbox = mailboxItem.value;
		String mailboxUid = mailboxItem.uid;
		String mailboxName = mailboxItem.value.name;
		String filename = String.format("%s@%s.json", login, domain.value.defaultAlias);
		Path outputPath = Paths.get(mailWorker.workingFolder.toAbsolutePath().toString(),
				String.format("%s@%s.json", mailboxName, domain.value.defaultAlias));
		return new SdsMailboxRecord(login, mailbox, mailboxUid, mailboxName, dataLocation, filename, outputPath);
	}

	@Override
	public MailboxIndexJson backupSdsMailbox(DirEntry de, ItemValue<Domain> domain, boolean downloadEmailContent) {
		try {
			logger.info("backup single '{}' {} ({})", de.kind.getDataprotectPrefix(), de.entryUid,
					de.email != null ? de.email : de.displayName);

			SdsMailboxRecord recordMailbox = de.kind == Kind.MAILSHARE ? createForMailshare(de, domain)
					: createForMailbox(de, domain);

			ITags tagsApi = mailWorker.serviceProvider.instance(ITags.class, ITagUids.defaultTags(de.entryUid));
			ISdsSyncStore productionStore = mailWorker.productionStores.get(recordMailbox.dataLocation);

			String subtreeUid = IMailReplicaUids.subtreeUid(domain.uid, de);
			IDbByContainerReplicatedMailboxes containerMailboxApi = mailWorker.serviceProvider
					.instance(IDbByContainerReplicatedMailboxes.class, subtreeUid);

			generateSdsMailboxJson(recordMailbox.outputPath, domain, recordMailbox.mailboxUid, recordMailbox.login,
					recordMailbox.mailbox, containerMailboxApi.all(), tagsApi.all(), productionStore,
					downloadEmailContent);
			return new MailboxIndexJson(recordMailbox.filename, recordMailbox.mailboxUid, domain.uid);
		} catch (Exception e) {
			mailWorker.logError(e, "Unable to backup '{}' {} ({})", de.kind.getDataprotectPrefix(), de.entryUid,
					de.email != null ? de.email : de.displayName);
		}
		return null;
	}

	private void generateSdsMailboxJson(Path outputPath, ItemValue<Domain> domain, String mailboxUid, String userLogin,
			Mailbox mailbox, List<ItemValue<MailboxFolder>> folders, List<ItemValue<Tag>> tags,
			ISdsSyncStore productionStore, boolean downloadEmailContent) throws Exception {

		Path outputTempPath = outputPath.resolveSibling(outputPath.getFileName().toString().replace(".json", ".tmp"));

		try (OutputStream outStream = new BufferedOutputStream(
				Files.newOutputStream(outputTempPath, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING));
				JsonGenerator generator = new JsonFactory().createGenerator(outStream, JsonEncoding.UTF8)) {
			Map<String, CyrusSdsBackupFolder> lastFoldersByUid = readFromLastBackup(mailboxUid);

			generator.writeStartObject();
			generator.writeNumberField("version", 2);
			generator.writeStringField("kind", mailbox.type.name());
			generator.writeStringField("mailboxUid", mailboxUid);
			generator.writeStringField("login", userLogin);
			generator.writeStringField("domainUid", domain.uid);
			generator.writeStringField("domainName", domain.value.defaultAlias);
			generator.writeStringField("dataLocation", mailbox.dataLocation);
			if (tags != null && !tags.isEmpty()) {
				generator.writeArrayFieldStart("tags");
				for (var t : tags) {
					generator.writeStartObject();
					generator.writeStringField("label", t.value.label);
					generator.writeStringField("color", t.value.color);
					generator.writeEndObject();
				}
				generator.writeEndArray();
			}
			generator.writeObjectFieldStart("backingstore");
			generator.writeStringField("archivekind", mailWorker.sysconf.stringValue(SysConfKeys.archive_kind.name()));
			generator.writeStringField("bucket", mailWorker.sysconf.stringValue(SysConfKeys.sds_s3_bucket.name()));
			generator.writeStringField("region", mailWorker.sysconf.stringValue(SysConfKeys.sds_s3_region.name()));
			generator.writeStringField("endpoint", mailWorker.sysconf.stringValue(SysConfKeys.sds_s3_endpoint.name()));
			generator.writeStringField("insecure", mailWorker.sysconf.stringValue(SysConfKeys.sds_s3_insecure.name()));
			generator.writeStringField("useSplitPath",
					mailWorker.sysconf.stringValue(SysConfKeys.sds_s3_split_path.name()));
			generator.writeEndObject();

			generator.writeArrayFieldStart("folders");
			for (ItemValue<MailboxFolder> folder : folders) {
				generateFolderEntryJson(userLogin, productionStore, downloadEmailContent, generator, lastFoldersByUid,
						folder);
			}
			generator.writeEndArray();
			generator.writeEndObject();

			moveFile(outputPath, outputTempPath);
		} catch (IOException e1) {
			mailWorker.logError(e1, "Unable to write {} for mailbox {}", outputTempPath, mailboxUid);
		}
	}

	private void generateFolderEntryJson(String userLogin, ISdsSyncStore productionStore, boolean downloadEmailContent,
			JsonGenerator generator, Map<String, CyrusSdsBackupFolder> lastFoldersByUid,
			ItemValue<MailboxFolder> folder) throws Exception {
		IDbMailboxRecords recordsApi = null;
		try {
			recordsApi = mailWorker.serviceProvider.instance(IDbMailboxRecords.class, folder.uid);
			if (recordsApi != null) {
				generator.writeStartObject();
				generator.writeStringField("uid", folder.uid);
				generator.writeStringField("fullName", folder.value.fullName);
				generator.writeStringField("name", folder.value.name);
				CyrusSdsBackupFolder lastFolder = lastFoldersByUid.get(folder.uid);
				generateSdsFolderContent(lastFolder, folder, generator, productionStore, recordsApi,
						downloadEmailContent);
				generator.writeEndObject();
			} else {
				throw new ServerFault("", ErrorCode.NOT_FOUND);
			}
		} catch (Exception ex) {
			if (ex instanceof ServerFault sf && ErrorCode.NOT_FOUND.equals(sf.getCode())) {
				mailWorker.logError(ex, "Unable to backup user {} folder {}: folder uid={} not found", userLogin,
						folder.value.fullName, folder.uid);
			} else {
				throw ex;
			}
		}
	}

	private Map<String, CyrusSdsBackupFolder> readFromLastBackup(String mailboxUid) {
		Map<String, CyrusSdsBackupFolder> lastFoldersByUid = new HashMap<>();

		try {
			if (mailWorker.lastBackup != null) {
				CyrusSdsBackupMailbox lastMailbox = mailWorker.lastBackup.getMailboxFromPreviousBackup(mailboxUid);
				if (lastMailbox != null) {
					lastFoldersByUid = lastMailbox.getFolders().stream()
							.collect(Collectors.toMap(f -> f.uid(), Function.identity()));
				}
			}
		} catch (Exception e) {
			mailWorker.logError(e, "Not able to retrieve previous backup mailbox state for uid {}", mailboxUid);
		}
		return lastFoldersByUid;
	}

	private void moveFile(Path outputPath, Path outputTempPath) {
		try {
			if (Files.exists(outputTempPath)) {
				Files.move(outputTempPath, outputPath, StandardCopyOption.REPLACE_EXISTING);
			}
		} catch (IOException e) {
			mailWorker.logError(e, "Unable to move {} to {}", outputTempPath, outputPath);
		}
	}

	private boolean isFolderHasBeenRegenerated(CyrusSdsBackupFolder lastFolder, ItemValue<MailboxFolder> folder) {
		if (folder.created == null || lastFolder.created() == null) {
			return false;
		}
		DateTimeFormatter fmt = CyrusSdsBackupFolder.DATE_FORMAT;
		return !Objects.equals(fmt.format(folder.created.toInstant()), fmt.format(lastFolder.created().toInstant()));
	}

	private boolean shouldBackupBeRebuild(CyrusSdsBackupFolder lastFolder, ItemValue<MailboxFolder> folder) {
		boolean isFirstBackup = lastFolder == null;
		boolean isOldBackupFormat = !isFirstBackup && (lastFolder.version() <= 0
				|| lastFolder.messages().stream().anyMatch(message -> message.itemId <= 0));
		boolean itemIdHasBeenRegenerated = !isFirstBackup && isFolderHasBeenRegenerated(lastFolder, folder);

		return isFirstBackup || isOldBackupFormat || itemIdHasBeenRegenerated;
	}

	private void generateSdsFolderContent(CyrusSdsBackupFolder lastFolder, ItemValue<MailboxFolder> folder,
			JsonGenerator generator, ISdsSyncStore productionStore, IDbMailboxRecords recordsApi,
			boolean downloadEmailContent) throws IOException {
		long changesetVersion = 0;
		Stream<CyrusSdsBackupMessage> lastBackupMessages = Stream.empty();
		if (!shouldBackupBeRebuild(lastFolder, folder)) {
			changesetVersion = lastFolder.version();
			lastBackupMessages = lastFolder.messages().stream();
		}

		ContainerChangeset<Long> changeSet = recordsApi.changesetById(changesetVersion);

		Stream<CyrusSdsBackupMessage> createdMessages = getCreatedMessages(recordsApi, changeSet);
		Stream<CyrusSdsBackupMessage> updatedMessages = getUpdatedMessages(recordsApi, changeSet);

		Stream<CyrusSdsBackupMessage> temp = Stream.concat(removeDeletedMessage(lastBackupMessages, changeSet),
				createdMessages);
		Stream<CyrusSdsBackupMessage> backupMessage = Stream.concat(removeUpdatedMessage(temp, changeSet),
				updatedMessages);

		generator.writeStringField("created", CyrusSdsBackupFolder.DATE_FORMAT.format(folder.created.toInstant()));
		generator.writeNumberField("changeSetVersion", changeSet.version);
		generator.writeArrayFieldStart("messages");

		backupMessage.forEach((CyrusSdsBackupMessage field) -> {
			// This is just a safety against broken databases, not encountered in real life
			if (field.guid != null && !field.guid.isEmpty()) {
				generateJson(generator, field);
				if (downloadEmailContent) {
					cliExportExtraAction(productionStore, field.guid);
				}
			}
		});
		generator.writeEndArray();
	}

	private Stream<CyrusSdsBackupMessage> getCreatedMessages(IDbMailboxRecords recordsApi,
			ContainerChangeset<Long> changeSet) {
		return Lists.partition(changeSet.created, 1000).stream() //
				.map(recordsApi::slice) //
				.flatMap(Collection::stream)
				.map(irecord -> new CyrusSdsBackupMessage(irecord.value.messageBody, irecord.value.internalDate,
						irecord.itemId, irecord.value.flags.stream().map(f -> f.flag).collect(Collectors.toSet())));
	}

	private Stream<CyrusSdsBackupMessage> getUpdatedMessages(IDbMailboxRecords recordsApi,
			ContainerChangeset<Long> changeSet) {
		return Lists.partition(changeSet.updated, 1000).stream() //
				.map(recordsApi::slice) //
				.flatMap(Collection::stream)
				.map(irecord -> new CyrusSdsBackupMessage(irecord.value.messageBody, irecord.value.internalDate,
						irecord.itemId, irecord.value.flags.stream().map(f -> f.flag).collect(Collectors.toSet())));
	}

	private Stream<CyrusSdsBackupMessage> removeDeletedMessage(Stream<CyrusSdsBackupMessage> lastBackupMessages,
			ContainerChangeset<Long> changeSet) {
		HashSet<Long> deletedItemsIds = new HashSet<>(changeSet.deleted);
		return lastBackupMessages.filter(msg -> !deletedItemsIds.contains(msg.itemId));
	}

	private Stream<CyrusSdsBackupMessage> removeUpdatedMessage(Stream<CyrusSdsBackupMessage> lastBackupMessages,
			ContainerChangeset<Long> changeSet) {
		HashSet<Long> updatedItemsIds = new HashSet<>(changeSet.updated);
		return lastBackupMessages.filter(msg -> !updatedItemsIds.contains(msg.itemId));
	}

	private void generateJson(JsonGenerator generator, CyrusSdsBackupMessage field) {
		try {
			generator.writeStartObject();
			generator.writeStringField("g", field.guid);
			generator.writeStringField("d", CyrusSdsBackupFolder.DATE_FORMAT.format(field.date.toInstant()));
			generator.writeNumberField("i", field.itemId);
			if (field.flags != null && !field.flags.isEmpty()) {
				generator.writeArrayFieldStart("f");
				field.flags.forEach(f -> {
					try {
						generator.writeString(f);
					} catch (IOException e) {
						mailWorker.ctx.error(e, "Unable to generate json data for message_body_guid {}", field.guid);
					}
				});
				generator.writeEndArray();
			}
			generator.writeEndObject();
		} catch (IOException ie) {
			mailWorker.ctx.error(ie, "Unable to generate json data for message_body_guid {}", field.guid);
		}
	}

	private void cliExportExtraAction(ISdsSyncStore productionStore, String guid) {
		if (productionStore != null && mailWorker.backupStore != null) {
			Path fp = mailWorker.backupStore.livePath(guid);
			if (!fp.getParent().toFile().exists()) {
				try {
					Files.createDirectories(fp.getParent());
				} catch (IOException ie) {
					throw new ServerFault(
							"Unable to create directory " + fp.getParent() + ", error message: " + ie.getMessage());
				}
			}
			SdsResponse response = productionStore.downloadRaw(GetRequest.of("", guid, fp.toString()));
			if (!response.succeeded()) {
				mailWorker.ctx.warn(String.format("unable to download guid %s: %s", guid, response));
			}
		}
	}

}
