/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.service.internal;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.function.Supplier;

import com.google.common.base.Suppliers;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerUid;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.repository.IContainerRouteStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.rest.BmContext;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.repository.provider.RepositoryProvider;

/**
 * For shared mailboxes, we redirect all rbac checks to use its mailbox
 * permissions.
 * 
 */
public class RbacRedirector {

	private RbacRedirector() {
	}

	/**
	 * For shared mailboxes, we redirect all rbac checks to use its mailbox
	 * permissions.
	 * 
	 * @param ctx  context
	 * @param cont container to check
	 * @return Supplier {@link IComposeRbacManager}
	 */
	public static Supplier<IComposeRbacManager> forContainer(BmContext ctx, Container cont) {
		return Suppliers.memoize(() -> forContainerImpl(ctx, cont.domainUid, cont.owner, cont.uid));
	}

	/**
	 * For shared mailboxes, we redirect all rbac checks to use its mailbox
	 * permissions.
	 * 
	 * @param ctx  context
	 * @param cont container to check
	 * @return Supplier {@link IComposeRbacManager}
	 */
	public static Supplier<IComposeRbacManager> forContainer(BmContext ctx, BaseContainerDescriptor cont) {
		return Suppliers.memoize(() -> forContainerImpl(ctx, cont.domainUid, cont.owner, cont.uid));
	}

	private static IComposeRbacManager forContainerImpl(BmContext ctx, String domainUid, String owner,
			String containerUid) {
		final RBACManager defaultRbacManager = RBACManager.forContext(ctx).forContainer(containerUid);
		DirEntry de = ctx.provider().instance(IDirectory.class, domainUid).findByEntryUid(owner);
		if (de != null && de.kind == Kind.SHARED_MAILBOX) {
			IContainerRouteStore router = RepositoryProvider.instance(IContainerRouteStore.class, ctx);
			containerUid = IMailboxAclUids.uidForMailbox(owner);
			DataLocation loc = router.routeOf(ContainerUid.of(containerUid));
			IContainerStore contStore = RepositoryProvider.instance(IContainerStore.class, ctx, loc);
			Container aclContainer;
			try {
				aclContainer = contStore.get(containerUid);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
			if (aclContainer == null) {
				throw ServerFault.notFound("Can't find container " + containerUid + " at location " + loc.serverUid());
			}

			return new ComposedRbacManager(
					Arrays.asList(defaultRbacManager, RBACManager.forContext(ctx).forContainer(aclContainer)));
		}
		return defaultRbacManager;
	}

}
