/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.container.service.internal;

import static net.bluemind.core.container.service.internal.ReadOnlyMode.checkWritable;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.base.MoreObjects;
import com.google.common.base.Suppliers;

import net.bluemind.core.api.DataSourceType;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.core.container.api.Count;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.CountFastPath;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.repository.IChangelogStore;
import net.bluemind.core.container.repository.IChangelogStore.LogEntry;
import net.bluemind.core.container.repository.ICustomDecorator;
import net.bluemind.core.container.repository.IItemStore;
import net.bluemind.core.container.repository.IItemValueStore;
import net.bluemind.core.container.repository.IWeightProvider;
import net.bluemind.core.container.service.IContainerStoreService;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.tx.wrapper.TxEnabler;
import net.bluemind.directory.api.ReservedIds;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;
import net.bluemind.tx.outbox.api.ITxOutbox;

public class ContainerStoreService<T> implements IContainerStoreService<T> {

	@FunctionalInterface
	public interface ToRun<R> {
		R execute() throws SQLException;
	}

	public <W> W atomic(ToRun<W> op) {
		return runner.on(op);
	}

	private <W> W atomicImpl(ToRun<W> op) {
		return TxEnabler.atomically(() -> doOrFail(op));
	}

	public <W> W doOrFail(ToRun<W> op) {
		return ServerFault.onException(op::execute, ErrorCode.SQL_ERROR);
	}

	private static interface AtomicRunner {
		<W> W on(ToRun<W> op);
	}

	protected static final Logger logger = LoggerFactory.getLogger(ContainerStoreService.class);
	protected final Container container;
	protected final String containerCacheKey;
	// fucking directory overrides that
	protected IItemStore itemStore;
	protected IItemValueStore<T> itemValueStore;
	protected IChangelogStore changelogStore;
	private AclService aclService;
	private String origin;
	protected SecurityContext securityContext;
	protected boolean hasChangeLog = true;
	private final IItemFlagsProvider<T> flagsProvider;
	private final IWeightSeedProvider<T> weightSeedProvider;
	private final IWeightProvider weightProvider;
	private final Supplier<ContainerChangeEventProducer> containerChangeEventProducer;
	private final Supplier<ITxOutbox> txOutbox;
	private final Supplier<Optional<ItemValueAuditLogService<T>>> logServiceSupplier;
	private final AtomicRunner runner;

	public final ReservedIds.ConsumerHandler doNothingOnIdsReservation = callback -> callback.accept(null);

	private static final Cache<String, Long> lastKnownVersions = Caffeine.newBuilder()
			.expireAfterWrite(5, TimeUnit.MINUTES).recordStats().build();

	public static class EmptyChangesetReg implements ICacheRegistration {

		@Override
		public void registerCaches(CacheRegistry cr) {
			cr.register("known.container.versions", lastKnownVersions);
		}

	}

	public static interface IItemFlagsProvider<W> {
		Collection<ItemFlag> flags(W value);
	}

	public static interface IWeightSeedProvider<W> {

		long weightSeed(W value);

	}

	private static final Collection<ItemFlag> UNFLAGGED = EnumSet.noneOf(ItemFlag.class);

	public ContainerStoreService(BmContext context, Container container, IItemValueStore<T> itemValueStore,
			ItemValueAuditLogService<T> logService) {
		this(context, container, itemValueStore, v -> UNFLAGGED, v -> 0L, seed -> seed, logService);
	}

	public ContainerStoreService(BmContext context, Container container, IItemValueStore<T> itemValueStore) {
		this(context, container, itemValueStore, v -> UNFLAGGED, v -> 0L, seed -> seed, null);
	}

	public ContainerStoreService(BmContext context, Container container, IItemValueStore<T> itemValueStore,
			IItemFlagsProvider<T> fProv, IWeightSeedProvider<T> wsProv, IWeightProvider wProv) {
		this(context, container, itemValueStore, fProv, wsProv, wProv, null);
	}

	public ContainerStoreService(BmContext context, Container container, IItemValueStore<T> itemValueStore,
			IItemFlagsProvider<T> fProv, IWeightSeedProvider<T> wsProv, IWeightProvider wProv,
			ItemValueAuditLogService<T> logService) {
		this.runner = context.getStorageFlavor() == DataSourceType.POSTGRESQL ? this::atomicImpl : this::doOrFail;
		securityContext = context.getSecurityContext();
		this.container = container;
		this.containerCacheKey = container.uid + "#" + container.id;
		this.origin = securityContext.getOrigin();
		this.itemStore = RepositoryProvider.instance(IItemStore.class, context, container);
		this.changelogStore = RepositoryProvider.instance(IChangelogStore.class, context, container);
		this.itemValueStore = itemValueStore;
		this.aclService = new AclService(context, securityContext, container);
		this.flagsProvider = fProv;
		this.weightSeedProvider = wsProv;
		this.weightProvider = wProv;
		this.logServiceSupplier = () -> {
			if (logService != null && StateContext.getState().equals(SystemState.CORE_STATE_RUNNING)) {
				return Optional.of(logService);
			} else {
				return Optional.empty();
			}
		};

		this.txOutbox = Suppliers.memoize(() -> context.provider().instance(ITxOutbox.class, container.domainUid,
				container.owner, container.type, container.uid, DataLocation.directory().serverUid()));

		this.containerChangeEventProducer = Suppliers
				.memoize(() -> new ContainerChangeEventProducer(securityContext, VertxPlatform.eventBus(), container));
	}

	protected ITxOutbox outbox() {
		return txOutbox.get();
	}

	public ContainerStoreService<T> withoutChangelog() {
		hasChangeLog = false;
		return this;
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(getClass()).add("items", itemValueStore).add("cont", container).toString();
	}

	private void assertChangeLog() {
		if (!hasChangeLog) {
			throw new ServerFault("no changelog for this container");
		}
	}

	public Count count(ItemFlagFilter filter) {
		Optional<CountFastPath> fastPath = filter.availableFastPath();
		if (fastPath.isPresent()) {
			return itemStore.fastpathCount(fastPath.get()).orElseGet(() -> {
				try {
					return Count.of(itemStore.count(filter));
				} catch (SQLException e) {
					throw ServerFault.sqlFault(e);
				}
			});
		}
		try {
			return Count.of(itemStore.count(filter));
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	protected void invalidateVersionCache() {
		lastKnownVersions.invalidate(containerCacheKey);
	}

	private <W> ContainerChangeset<W> cacheIfUnchanged(long from, ToRun<ContainerChangeset<W>> op) {
		Long maybeSameAsSince = lastKnownVersions.getIfPresent(containerCacheKey);
		if (maybeSameAsSince != null && maybeSameAsSince.longValue() <= from) {
			return ContainerChangeset.empty(maybeSameAsSince.longValue());
		}
		ContainerChangeset<W> cs = doOrFail(op);
		if (isValidContainerVersion(cs.version)) {
			lastKnownVersions.put(containerCacheKey, cs.version);
		} else {
			if (maybeSameAsSince != null) {
				return ContainerChangeset.empty(maybeSameAsSince.longValue());
			}
			cs.version = sanitizeContainerVersion(from);
		}
		return cs;
	}

	private boolean isValidContainerVersion(long version) {
		return version >= 0;
	}

	private long sanitizeContainerVersion(long version) {
		return Math.max(version - 1, 0);
	}

	public ContainerChangeset<String> changeset(Long from, long to) {
		assertChangeLog();
		final long since = null == from ? 0L : from;
		return cacheIfUnchanged(since, () -> changelogStore.changeset(weightProvider, since, to));
	}

	public ContainerChangeset<Long> changesetById(Long from, long to) {
		assertChangeLog();
		final long since = null == from ? 0L : from;
		return cacheIfUnchanged(since, () -> changelogStore.changesetById(weightProvider, since, to));
	}

	public ContainerChangeset<ItemIdentifier> fullChangesetById(Long from, long to) {
		assertChangeLog();
		final long since = null == from ? 0L : from;
		return cacheIfUnchanged(since, () -> changelogStore.fullChangesetById(weightProvider, since, to));
	}

	public ContainerChangeset<ItemVersion> changesetById(long from, ItemFlagFilter filter) {
		assertChangeLog();
		return cacheIfUnchanged(from, () -> changelogStore.changesetById(weightProvider, from, Long.MAX_VALUE, filter));
	}

	@Override
	public ItemValue<T> get(String uid, Long version) {
		return doOrFail(() -> {
			Item item = itemStore.get(uid);
			if (version != null && item != null && item.version != version) {
				logger.warn("call get with version and version are different : expected {} actual {}", version,
						item.version);
			}
			return getItemValue(item);
		});
	}

	@Override
	public ItemValue<T> getLight(String uid, Long version) {
		return doOrFail(() -> {
			Item item = itemStore.get(uid);
			if (version != null && item != null && item.version != version) {
				logger.warn("call get with version and version are different : expected {} actual {}", version,
						item.version);
			}
			return getItemValueLight(item);
		});
	}

	@Override
	public ItemValue<T> get(long id, Long version) {
		return doOrFail(() -> {
			Item item = itemStore.getById(id);
			if (version != null && item != null && item.version != version) {
				logger.warn("call get with version and version are different : expected {} actual {}", version,
						item.version);
			}

			return getItemValue(item);
		});

	}

	protected ItemValue<T> getItemValue(Item item) {
		if (item == null) {
			return null;
		}
		T value = getValue(item);
		if (value == null && !itemValueStore.toString().equals("UserSettingsStore")) {
			logger.warn("null value for existing item {} with store {}", item, itemValueStore);
		}
		ItemValue<T> ret = ItemValue.create(item, value);
		decorate(item, ret);
		return ret;
	}

	protected ItemValue<T> getItemValueLight(Item item) {
		if (item == null) {
			return null;
		}
		T value = getValueLight(item);
		if (value == null && !itemValueStore.toString().equals("UserSettingsStore")) {
			logger.warn("null value for existing item {} with store {}", item, itemValueStore);
		}
		return ItemValue.create(item, value);
	}

	@Override
	public ItemValue<T> getByExtId(String extId) {
		return doOrFail(() -> {
			Item item = itemStore.getByExtId(extId);
			if (item == null) {
				return null;
			}

			return getItemValue(item);
		});
	}

	@Override
	public String getUidByExtId(String extId) {
		return doOrFail(() -> {
			Item item = itemStore.getByExtId(extId);
			if (item == null) {
				return null;
			}
			return item.uid;
		});
	}

	protected T getValue(Item item) {
		try {
			return itemValueStore.get(item);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	protected T getValueLight(Item item) {
		try {
			return itemValueStore.getLight(item);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public ItemVersion create(String uid, String displayName, T value) {
		return create(uid, null, displayName, value);
	}

	@Override
	public ItemVersion create(String uid, String extId, String displayName, T value) {
		return createWithId(uid, null, extId, displayName, value);
	}

	protected ItemVersion createWithId(String uid, Long internalId, String extId, String displayName, T value, // NOSONAR
			IChangelogStore changelogStore, IItemStore itemStore, IItemValueStore<T> itemValueStore) {
		Item item = new Item();
		item.uid = uid;
		item.externalId = extId;
		if (internalId != null) {
			item.id = internalId;
		}
		item.displayName = displayName;
		item.flags = flagsProvider.flags(value);
		return create(item, value, changelogStore, itemStore, itemValueStore, doNothingOnIdsReservation);
	}

	public Item flagged(Item cur, T value) {
		cur.flags = flagsProvider.flags(value);
		return cur;
	}

	@Override
	public ItemVersion createWithId(String uid, Long internalId, String extId, String displayName, T value) {
		return createWithId(uid, internalId, extId, displayName, value, changelogStore, itemStore, itemValueStore);
	}

	@Override
	public ItemVersion create(Item item, T value) {
		return create(item, value, changelogStore, itemStore, itemValueStore, doNothingOnIdsReservation);
	}

	protected ItemVersion create(Item item, T value, ReservedIds.ConsumerHandler handler) {
		return create(item, value, changelogStore, itemStore, itemValueStore, handler);
	}

	private ItemVersion create(Item item, T value, IChangelogStore changelogStore, IItemStore itemStore,
			IItemValueStore<T> itemValueStore, ReservedIds.ConsumerHandler handler) {
		checkWritable();

		String uid = item.uid;
		Long internalId = item.id;
		return atomic(() -> {
			Item created;
			try {
				created = itemStore.create(item);
			} catch (SQLException e) {
				logger.error(e.getMessage(), e);
				throw ServerFault.alreadyExists("entry[" + uid + " - " + internalId + "]@" + container.uid
						+ " already exists (" + e.getMessage() + ")");
			}
			if (created == null) {
				throw new ServerFault(
						"itemStore " + itemStore + " has **NOT** created item " + item + " can't continue");
			}

			createValue(created, value, itemValueStore);
			if (hasChangeLog) {
				changelogStore.itemCreated(LogEntry.create(created.version, created.uid, created.externalId,
						securityContext.getSubject(), origin, created.id, weightSeedProvider.weightSeed(value)));
			}
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			if (hasChangeLog) {
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}

			ItemValue<T> iv = ItemValue.create(created, value);
			beforeCreationInBackupStore(iv);

			handler.acceptConsumer(reservedIds -> txOutbox.get().forKafka(iv, reservedIds, false));

			TxEnabler.durableStorageAction(
					() -> logServiceSupplier.get().ifPresent(logService -> logService.logCreate(iv)));

			return created.itemVersion();
		});
	}

	protected void beforeCreationInBackupStore(@SuppressWarnings("unused") ItemValue<T> itemValue) {
		// This methode can be override in child class to perform an operation before
		// backuping a creation
	}

	@Override
	public void attach(String uid, String displayName, T value) {
		atomic(() -> {

			Item item = itemStore.get(uid);
			if (item == null) {
				item = new Item();
				item.uid = uid;
				item.displayName = displayName;
				item = itemStore.create(item);
			} else {
				item = itemStore.touch(uid);
			}

			if (hasChangeLog) {
				changelogStore.itemUpdated(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), origin, item.id, weightSeedProvider.weightSeed(value)));
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			createValue(item, value);
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			return null;
		});
	}

	private void createValue(Item item, T value) throws SQLException {
		createValue(item, value, itemValueStore);
	}

	protected void createValue(Item item, T value, IItemValueStore<T> itemValueStore) throws SQLException {
		itemValueStore.create(item, value);
	}

	@Override
	public ItemVersion update(String uid, String displayName, T value) {
		Item item = new Item();
		item.uid = uid;
		return update(item, displayName, value);
	}

	@Override
	public ItemVersion update(Item item, String displayName, T value) {
		return update(item, displayName, value, doNothingOnIdsReservation);
	}

	protected ItemVersion update(Item item, String displayName, T value, ReservedIds.ConsumerHandler handler) {
		checkWritable();

		return atomic(() -> {

			String dnToApply = displayName;
			if (dnToApply == null) {
				// try to preserve the existing display name
				Item existing = itemStore.get(item.uid);
				if (existing == null) {
					throw ServerFault.notFound("entry[" + item.uid + "]@" + container.uid + " not found ");
				}

				dnToApply = existing.displayName;
			}
			Item updated = itemStore.update(item, dnToApply, flagsProvider.flags(value));
			if (updated == null) {
				throw ServerFault.notFound("entry[uid: " + item.uid + " / id:" + item.id + "]@" + container.uid
						+ " not found, dn: " + dnToApply);
			}

			if (hasChangeLog) {
				changelogStore.itemUpdated(LogEntry.create(updated.version, updated.uid, updated.externalId,
						securityContext.getSubject(), origin, updated.id, weightSeedProvider.weightSeed(value)));
			}

			T oldValue = doOrFail(() -> itemValueStore.get(updated));
			updateValue(updated, value);
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			if (hasChangeLog) {
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}

			ItemValue<T> itemValue = ItemValue.create(updated, value);
			handler.acceptConsumer(reservedIds -> txOutbox.get().forKafka(itemValue, reservedIds, false));

			TxEnabler.durableStorageAction(() -> logServiceSupplier.get().ifPresent(logService -> {
				if (updated.flags.contains(ItemFlag.Deleted)) {
					logService.logDelete(itemValue);
				} else {
					logService.logUpdate(itemValue, oldValue);
				}
			}));

			return updated.itemVersion();
		});
	}

	@Override
	public ItemVersion update(long itemId, String displayName, T value) {
		checkWritable();

		return atomic(() -> {

			String dnToApply = displayName;
			if (dnToApply == null) {
				// try to preserve the existing display name
				Item existing = itemStore.getById(itemId);
				if (existing == null) {
					throw ServerFault.notFound("entry[id: " + itemId + "]@" + container.uid + " not found");
				}

				dnToApply = existing.displayName;
			}

			Item item = itemStore.update(itemId, dnToApply, flagsProvider.flags(value));
			if (item == null) {
				throw ServerFault.notFound("entry[id: " + itemId + "]@" + container.uid + " not found ");
			}
			if (hasChangeLog) {
				changelogStore.itemUpdated(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), origin, item.id, weightSeedProvider.weightSeed(value)));
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			T oldValue = doOrFail(() -> itemValueStore.get(item));
			updateValue(item, value);
			ItemValue<T> iv = ItemValue.create(item, value);

			TxEnabler.durableStorageAction(() -> logServiceSupplier.get().ifPresent(logService -> {
				if (item.flags.contains(ItemFlag.Deleted)) {
					logService.logDelete(iv);
				} else {
					logService.logUpdate(iv, oldValue);
				}
			}));
			TxEnabler.durableStorageAction(this::invalidateVersionCache);

			txOutbox.get().forKafka(iv, null, false);
			return item.itemVersion();
		});
	}

	protected void preUpdateValue(Item newItem, T newValue, Supplier<T> oldValue) throws SQLException {
		// override if necessary
	}

	protected void updateValue(Item item, T value) throws SQLException {
		preUpdateValue(item, value, () -> doOrFail(() -> itemValueStore.get(item)));
		itemValueStore.update(item, value);
	}

	@Override
	public ItemVersion delete(String uid) {
		checkWritable();
		Optional<ItemValueAuditLogService<T>> logSupplier = logServiceSupplier.get();
		boolean needsPushValue = logSupplier.isPresent() || !txOutbox.get().isPaused();

		return atomic(() -> {
			if (itemStore.getItemId(uid) == null) {
				return null;
			}
			Item item = itemStore.touch(uid);
			ItemValue<T> itemValue;
			if (needsPushValue) {
				itemValue = getItemValue(item);
			} else {
				itemValue = null;
			}
			deleteValue(item);
			if (hasChangeLog) {
				changelogStore.itemDeleted(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), origin, item.id, 0L));
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			itemStore.delete(item);
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			ContainerDescriptor cd = ContainerDescriptor.create(container.uid, container.name, container.owner,
					container.type, container.domainUid, false);
			cd.internalId = container.id;
			if (needsPushValue) {
				txOutbox.get().forKafka(itemValue, null, true);
				TxEnabler.durableStorageAction(
						() -> logSupplier.ifPresent(logService -> logService.logDelete(itemValue)));
			}
			return item.itemVersion();
		});
	}

	@Override
	public ItemVersion delete(long id) {
		checkWritable();
		Optional<ItemValueAuditLogService<T>> logSupplier = logServiceSupplier.get();
		boolean needsPushValue = logSupplier.isPresent() || !txOutbox.get().isPaused();

		return atomic(() -> {
			Item item = itemStore.getById(id);
			if (item == null) {
				return null;
			}

			item = itemStore.touch(item.uid);
			ItemValue<T> itemValue;
			if (needsPushValue) {
				itemValue = getItemValue(item);
			} else {
				itemValue = null;
			}

			deleteValue(item);
			if (hasChangeLog) {
				changelogStore.itemDeleted(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), origin, item.id, 0L));
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			itemStore.delete(item);
			TxEnabler.durableStorageAction(this::invalidateVersionCache);

			ContainerDescriptor cd = ContainerDescriptor.create(container.uid, container.name, container.owner,
					container.type, container.domainUid, false);
			cd.internalId = container.id;

			if (needsPushValue) {
				txOutbox.get().forKafka(itemValue, null, true);
				TxEnabler.durableStorageAction(
						() -> logServiceSupplier.get().ifPresent(logService -> logService.logDelete(itemValue)));
			}

			return item.itemVersion();
		});
	}

	@Override
	public void detach(String uid) {
		atomic(() -> {
			if (itemStore.getItemId(uid) == null) {
				return null;
			}
			Item item = itemStore.touch(uid);
			deleteValue(item);
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			if (hasChangeLog) {
				changelogStore.itemUpdated(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), origin, item.id, 0L));
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			return null;
		});
	}

	protected void deleteValue(Item item) throws SQLException {
		itemValueStore.delete(item);
	}

	@Override
	public void deleteAll() {
		checkWritable();

		atomic(() -> {
			// delete values
			deleteValues();
			// delete container
			if (hasChangeLog) {
				changelogStore.allItemsDeleted(securityContext.getSubject(), origin);
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			// delete items
			itemStore.deleteAll();
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			return null;
		});
	}

	@Override
	public void prepareContainerDelete() {
		checkWritable();

		atomic(() -> {
			// delete acl
			aclService.deleteAll();
			// delete values
			deleteValues();
			// delete changelog
			if (hasChangeLog) {
				changelogStore.deleteLog();
			}
			// delete items
			itemStore.deleteAll();
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			return null;
		});
	}

	protected void deleteValues() {
		try {
			itemValueStore.deleteAll();
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public IItemStore getItemStore() {
		return itemStore;
	}

	public IItemValueStore<T> getItemValueStore() {
		return itemValueStore;
	}

	private List<ItemValue<T>> getItemsValue(List<Item> items, List<? extends ICustomDecorator> customDecorator) {

		List<ItemValue<T>> ret = new ArrayList<>(items.size());

		List<T> values = null;
		try {
			values = customDecorator == null ? itemValueStore.getMultiple(items)
					: itemValueStore.getMultipleCustom(items, customDecorator);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
		List<Item> nonNullValues = new ArrayList<>(items.size());

		if (values.size() == items.size()) {
			Iterator<Item> itItems = items.iterator();
			Iterator<T> itValues = values.iterator();

			while (itItems.hasNext()) {
				Item item = itItems.next();
				T value = itValues.next();
				if (value != null) {
					ret.add(ItemValue.create(item, value));
					nonNullValues.add(item);
				}
			}

			decorate(nonNullValues, ret, customDecorator);
			return ret;
		} else {
			logger.warn("Mismatch in value and item count on container {}", container.uid);
			return getItemsValueByIndividualLookup(items);
		}
	}

	public List<ItemValue<T>> getItemsValue(List<Item> items) {
		return this.getItemsValue(items, null);
	}

	public List<ItemValue<T>> getItemsValueCustom(List<Item> items, List<? extends ICustomDecorator> customDecorator) {
		return this.getItemsValue(items, customDecorator);
	}

	private List<ItemValue<T>> getItemsValueByIndividualLookup(List<Item> items) {
		List<ItemValue<T>> ret = new ArrayList<>(items.size());

		for (Item item : items) {
			try {
				T value = itemValueStore.get(item);
				if (value != null) {
					ItemValue<T> itemValue = ItemValue.create(item, value);
					ret.add(itemValue);
					decorate(item, itemValue);
				}
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		}
		return ret;
	}

	protected void decorate(List<Item> items, List<ItemValue<T>> values,
			List<? extends ICustomDecorator> customDecorator) {
		Iterator<ItemValue<T>> it = values.iterator();
		for (Item item : items) {
			if (customDecorator != null) {
				decorate(item, it.next(), customDecorator);
			} else {
				decorate(item, it.next());
			}
		}
	}

	protected void decorate(List<Item> items, List<ItemValue<T>> values) {
		Iterator<ItemValue<T>> it = values.iterator();
		for (Item item : items) {
			decorate(item, it.next());
		}
	}

	protected void decorate(Item item, ItemValue<T> value, List<? extends ICustomDecorator> customDecorator) {
		decorate(item, value);
	}

	protected void decorate(Item item, ItemValue<T> value) {
		// OK
	}

	public List<ItemValue<T>> getMultiple(List<String> uids, List<? extends ICustomDecorator> customDecorator) {
		if (uids == null || uids.isEmpty()) {
			return Collections.emptyList();
		}
		return doOrFail(() -> {
			List<Item> items = null;

			try {
				items = itemStore.getMultiple(uids);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}

			return customDecorator == null ? getItemsValue(items) : getItemsValueCustom(items, customDecorator);
		});
	}

	public List<ItemValue<T>> getMultiple(List<String> uids) {
		return getMultiple(uids, null);
	}

	public List<ItemValue<T>> getMultipleCustom(List<String> uids, List<? extends ICustomDecorator> customDecorator) {
		return getMultiple(uids, customDecorator);
	}

	public List<ItemValue<T>> getMultipleById(List<Long> ids) {
		if (ids == null || ids.isEmpty()) {
			return Collections.emptyList();
		}
		return doOrFail(() -> {
			List<Item> items = null;

			try {
				items = itemStore.getMultipleById(ids);
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}

			return getItemsValue(items);
		});
	}

	public List<ItemValue<T>> all() {
		return doOrFail(() -> {
			List<Item> items = null;

			try {
				items = itemStore.all();
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}

			return getItemsValue(items);
		});
	}

	@Override
	public ItemVersion touch(String uid) {
		checkWritable();

		return atomic(() -> {
			Item item = itemStore.touch(uid);

			if (item == null) {
				throw ServerFault.notFound("entry[" + uid + "]@" + container.uid + " not found ");
			}
			TxEnabler.durableStorageAction(this::invalidateVersionCache);

			if (hasChangeLog) {
				T value = getValue(item);
				changelogStore.itemUpdated(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), origin, item.id, weightSeedProvider.weightSeed(value)));

				ItemValue<T> iv = ItemValue.create(item, value);
				txOutbox.get().forKafka(iv, null, false);
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			return item.itemVersion();
		});
	}

	@Override
	public List<String> allUids() {

		try {
			List<Item> items = itemStore.all();
			List<String> ret = new ArrayList<>(items.size());
			for (Item i : items) {
				ret.add(i.uid);
			}
			return ret;
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public List<Long> allIds() {
		try {
			return itemStore.allItemIds();
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public long getVersion() {
		return lastKnownVersions.get(containerCacheKey, k -> {
			try {
				return itemStore.getVersion();
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		});
	}

	@Override
	public long setExtId(String uid, String extId) {
		return atomic(() -> {
			Item item = itemStore.setExtId(uid, extId);

			if (item == null) {
				throw ServerFault.notFound("entry[" + uid + "]@" + container.uid + " not found");
			}
			TxEnabler.durableStorageAction(this::invalidateVersionCache);
			if (hasChangeLog) {
				T value = getValue(item);
				changelogStore.itemUpdated(LogEntry.create(item.version, item.uid, item.externalId,
						securityContext.getSubject(), origin, item.id, weightSeedProvider.weightSeed(value)));
				ItemValue<T> iv = ItemValue.create(item, value);
				txOutbox.get().forKafka(iv, null, false);
				TxEnabler.durableStorageAction(() -> containerChangeEventProducer.get().produceEvent());
			}
			return item.version;
		});
	}

	@Override
	public ItemValueExists exists(String uid) {
		try {
			Long itemId = itemStore.getItemId(uid);
			if (itemId == null) {
				return ItemValueExists.DOESNOTEXISTS;
			}
			Item item = Item.create(uid, itemId);
			return new ItemValueExists(true, itemValueStore.exists(item));
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public void restoreDelete(ItemVersion iv) {
		try {
			itemStore.touchContainer(iv);
			logger.info("Restore version for deletion v{} in {}", iv.version, container.uid);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	public void restore(ItemValue<T> item, boolean isCreate) {
		if (isCreate) {
			create(item.item(), item.value);
		} else {
			update(item.item(), item.displayName, item.value);
		}
	}

}
