/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.directory.service.internal.cql;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import com.google.common.collect.Lists;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.directory.cql.common.CqlContStoreService;
import net.bluemind.directory.repository.DirEntryNG;
import net.bluemind.domain.api.Domain;

public class CqlDirectoryService implements IDirectory {

	private final CqlContStoreService store;
	private final ItemValue<Domain> domain;

	public CqlDirectoryService(BmContext context, Container container, ItemValue<Domain> domainValue) {
		this.store = new CqlContStoreService(context, container, domainValue);
		this.domain = domainValue;
	}

	@Override
	public DirEntry findByEntryUid(String entryUid) throws ServerFault {
		return Optional.ofNullable(store.get(entryUid, null)).map(iv -> DirEntry.copyOf(iv.value)).orElse(null);
	}

	@Override
	public ItemValue<DirEntry> findItemValueByEntryUid(String entryUid) throws ServerFault {
		return Optional.ofNullable(store.get(entryUid, null)).map(iv -> ItemValue.create(iv, DirEntry.copyOf(iv.value)))
				.orElse(null);
	}

	@Override
	public DirEntry getByEmail(String email) {
		return Optional.ofNullable(store.byEmail(email)).map(it -> store.get(it.id, null)).map(iv -> iv.value)
				.orElse(null);
	}

	@Override
	public DirEntry getRoot() throws ServerFault {
		return Optional.ofNullable(store.get(domain.uid, null)).map(iv -> iv.value).orElse(null);
	}

	@Override
	public DirEntry getEntry(String path) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<DirEntry> getEntries(String path) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public TaskRef delete(String path) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public ItemValue<VCard> getVCard(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public TaskRef deleteByEntryUid(String entryUid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public ContainerChangeset<String> changeset(Long since) throws ServerFault {
		return store.changeset(since, Long.MAX_VALUE);
	}

	@Override
	public ListResult<ItemValue<DirEntry>> search(DirEntryQuery query) throws ServerFault {
		List<String> uids = searchUids(query).values;
		Predicate<ItemValue<DirEntryNG>> matched = getMatcher(query);
		ArrayList<ItemValue<DirEntry>> ret = new ArrayList<>(uids.size());
		for (List<String> slice : Lists.partition(uids, 256)) {
			store.getMultiple(slice).stream().filter(matched)
					.forEach(iv -> ret.add(ItemValue.create(iv, DirEntry.copyOf(iv.value))));
		}
		System.err.println("** DIRNG search -> " + ret.size() + " item(s)");
		return ListResult.create(ret);
	}

	@Override
	public ListResult<String> searchUids(DirEntryQuery query) throws ServerFault {
		return ListResult.create(store.allUids());
	}

	private Predicate<ItemValue<DirEntryNG>> getMatcher(DirEntryQuery query) {
		Set<Kind> kinds = query.kindsFilter != null && !query.kindsFilter.isEmpty() ? Set.copyOf(query.kindsFilter)
				: EnumSet.allOf(Kind.class);
		return iv -> {
			if (!kinds.contains(iv.value.kind)) {
				return false;
			}
			if (query.nameFilter != null && !iv.value.displayName.startsWith(query.nameFilter)) {
				return false;
			}
			if (query.systemFilter && iv.value.system) {
				return false;
			}
			if (query.hiddenFilter && iv.value.hidden) {
				return false;
			}

			return true;
		};
	}

	@Override
	public byte[] getEntryIcon(String entryUid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public byte[] getEntryPhoto(String entryUid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public byte[] getIcon(String path) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Set<String> getRolesForDirEntry(String entryUid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Set<String> getRolesForOrgUnit(String orgUnitUid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<ItemValue<DirEntry>> getByRoles(List<String> roles) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<ItemValue<DirEntry>> getMultiple(List<String> id) {
		return store.getMultiple(id).stream().map(iv -> ItemValue.<DirEntry>create(iv, DirEntry.copyOf(iv.value)))
				.toList();
	}

	@Override
	public List<ItemValue<DirEntry>> getMultipleLight(List<String> id) {
		return getMultiple(id);
	}

}
