/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.user.service.internal;

import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.NotImplementedException;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.user.api.ISharedMailbox;
import net.bluemind.user.api.SharedMailbox;
import net.bluemind.user.api.User;
import net.bluemind.user.service.IInCoreUser;

public class SharedMailboxService implements ISharedMailbox {

	private final IInCoreUser userService;

	public SharedMailboxService(IInCoreUser userService) {
		this.userService = userService;
	}

	@Override
	public SharedMailbox get(String uid) {
		return SharedMailbox.create(userService.get(uid));
	}

	@Override
	public ItemValueExists itemValueExists(String uid) {
		return userService.itemValueExists(uid);
	}

	@Override
	public ItemValue<SharedMailbox> getComplete(String uid) throws ServerFault {
		Optional<ItemValue<User>> opUserItem = Optional.ofNullable(userService.getComplete(uid));
		return opUserItem.map(user -> ItemValue.create(user.item(), SharedMailbox.create(user.value))).orElse(null);
	}

	@Override
	public ItemValue<SharedMailbox> getLight(String uid) throws ServerFault {
		Optional<ItemValue<User>> opUserItem = Optional.ofNullable(userService.getLight(uid));
		return opUserItem.map(user -> ItemValue.create(user.item(), SharedMailbox.create(user.value))).orElse(null);
	}

	@Override
	public ItemValue<SharedMailbox> byEmail(String email) throws ServerFault {
		Optional<ItemValue<User>> opUserItem = Optional.ofNullable(userService.byEmail(email));
		return opUserItem.map(user -> ItemValue.create(user.item(), SharedMailbox.create(user.value))).orElse(null);
	}

	@Override
	public ItemValue<SharedMailbox> byName(String name) throws ServerFault {
		Optional<ItemValue<User>> opUserItem = Optional.ofNullable(userService.byLogin(name));
		return opUserItem.map(user -> ItemValue.create(user.item(), SharedMailbox.create(user.value))).orElse(null);
	}

	@Override
	public TaskRef delete(String uid) throws ServerFault {
		return userService.delete(uid);
	}

	@Override
	public List<String> allUids() throws ServerFault {
		return userService.allUids(false);
	}

	@Override
	public VCard getVCard(String uid) throws ServerFault {
		return userService.getVCard(uid);
	}

	@Override
	public void restore(ItemValue<SharedMailbox> item, boolean isCreate) {
		throw new NotImplementedException("restore endpoint for shared mailbox is not implemented");
	}
}
