/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.ui.gwtsharing.client;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map.Entry;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.Optional;

import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.core.client.JsArrayString;

import net.bluemind.core.api.AsyncHandler;
import net.bluemind.core.commons.gwt.JsMapStringJsObject;
import net.bluemind.core.container.api.IContainerManagementAsync;
import net.bluemind.core.container.api.IContainerManagementPromise;
import net.bluemind.core.container.api.gwt.endpoint.ContainerManagementGwtEndpoint;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.directory.api.IDirectoryPromise;
import net.bluemind.directory.api.gwt.endpoint.DirectoryGwtEndpoint;
import net.bluemind.gwtconsoleapp.base.editor.gwt.IGwtModelHandler;
import net.bluemind.gwtconsoleapp.base.handler.DefaultAsyncHandler;
import net.bluemind.ui.common.client.forms.Ajax;

public abstract class BaseSharingModelHandler implements IGwtModelHandler {

	private final String modelId;
	private Map<String, List<AccessControlEntry>> entriesToUpdate;
	private List<AccessControlEntry> mailboxAclToUpdate;
	private final boolean isSharedMailbox;

	public BaseSharingModelHandler(String modelId) {
		this.modelId = modelId;
		this.entriesToUpdate = new HashMap<>();
		this.mailboxAclToUpdate = new ArrayList<>();
		this.isSharedMailbox = "shared-mailbox-sharing".equals(modelId);
	}

	@Override
	public void load(JavaScriptObject model, final AsyncHandler<Void> handler) {
		final JsMapStringJsObject m = model.cast();

		IContainerManagementPromise cm = new ContainerManagementGwtEndpoint(Ajax.TOKEN.getSessionId(),
				getContainerUid(model)).promiseApi();
		cm.getDescriptor().thenCompose(container -> {
			IDirectoryPromise directoryService = new DirectoryGwtEndpoint(Ajax.TOKEN.getSessionId(),
					container.domainUid).promiseApi();
			return directoryService.getEntry(container.ownerDirEntryPath);
		}).thenCompose(owner -> {
			SharingModel.init(m, modelId);
			SharingModel.populate(m, modelId, owner);
			return cm.getAccessControlList();
		}).thenAccept(aclFromDb -> {
			SharingModel.populate(m, modelId, aclFromDb);
			SharingModel.populateCache(m, modelId, aclFromDb);
			handler.success(null);
		}).exceptionally(t -> {
			handler.success(null);
			return null;
		});

	}

	@Override
	public void save(JavaScriptObject model, final AsyncHandler<Void> handler) {
		SharingModel currentSharingModel = SharingModel.get(model, modelId);	
		if (currentSharingModel != null) {
			List<AccessControlEntry> aclFromModel = Optional.ofNullable(currentSharingModel.getAcl()).map(l -> AccessControlEntry.expand(l)).orElse(null);
			List<AccessControlEntry> aclFromCache = Optional.ofNullable(currentSharingModel.getCachingAcl()).map(l -> AccessControlEntry.expand(l)).orElse(null);
			String containerUid = getContainerUid(model);
			this.mailboxAclToUpdate = new ArrayList<>();	
			if (!aclEquals(aclFromModel, aclFromCache)) {
				aclFromModel.addAll(aclFromCache.stream().filter(ace -> ace.subject.startsWith("x-calendar-p")).collect(Collectors.toSet()));
				this.mailboxAclToUpdate.addAll(aclFromModel);
			}
			
			JsArrayString sharingsIds = SharingsModel.getSharingsModelIds(model);
			if (sharingsIds != null) {
				for (int i = 0; i < sharingsIds.length(); i++) {
					String sharingId = sharingsIds.get(i);
					setAcl(model, sharingId, handler);
				}
			}

			if (this.mailboxAclToUpdate != null && !this.mailboxAclToUpdate.isEmpty()) {
				setAclForContainer(containerUid, this.mailboxAclToUpdate, handler);
			}
			for (Entry<String, List<AccessControlEntry>> entryToUpdate: this.entriesToUpdate.entrySet()) {
				if (!entryToUpdate.getValue().isEmpty()) {
					setAclForContainer(entryToUpdate.getKey(), entryToUpdate.getValue(), handler);
				}
			}
		}
		handler.success(null);
	}

	private void setAcl(JavaScriptObject model, String modelId, final AsyncHandler<Void> handler) {
		JsMapStringJsObject m = model.cast();
		SharingsModel sharingModel = m.get(modelId).cast();

		for (String containerUid : sharingModel.getContainers()) {
			List<AccessControlEntry> aclFromModel = Optional.ofNullable(sharingModel.getAcl(containerUid)).map(l -> AccessControlEntry.expand(l)).orElse(null);
			List<AccessControlEntry> aclFromCache = Optional.ofNullable(sharingModel.getCachingAcl(containerUid)).map(l -> AccessControlEntry.expand(l)).orElse(null);
			if (!aclEquals(aclFromModel, aclFromCache)) {
				aclFromModel.addAll(aclFromCache.stream().filter(ace -> ace.subject.startsWith("x-calendar-p")).collect(Collectors.toList()));
				this.entriesToUpdate.put(containerUid, aclFromModel);
			}
		}
	}
 
	private void setAclForContainer(String containerUid, List<AccessControlEntry> newTarget, final AsyncHandler<Void> handler) {
		new ContainerManagementGwtEndpoint(Ajax.TOKEN.getSessionId(),
			containerUid).setAccessControlList(newTarget, handler);
	}

	private boolean aclEquals(List<AccessControlEntry> aclFromModel, List<AccessControlEntry> aclFromCache) {
		return new HashSet<>(AccessControlEntry.compact(aclFromCache)).equals(new HashSet<>(AccessControlEntry.compact(aclFromModel)));
	}

	protected abstract String getContainerUid(JavaScriptObject model);
}
