package net.bluemind.indexing.incremental.todolist;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.persistence.DataSourceRouter;
import net.bluemind.core.rest.BmContext;
import net.bluemind.indexing.incremental.AbstractTypeIndexer;
import net.bluemind.indexing.incremental.AbstractTypeIndexer.IndexWriter;
import net.bluemind.indexing.incremental.TypeIndexerFactory;
import net.bluemind.lib.elasticsearch.ESearchActivator;
import net.bluemind.todolist.api.ITodoList;
import net.bluemind.todolist.api.ITodoUids;
import net.bluemind.todolist.api.VTodo;
import net.bluemind.todolist.persistence.VTodoIndexStore;

public class TodolistsIncrementalIndexFactory implements TypeIndexerFactory {
	private static final Logger logger = LoggerFactory.getLogger(TodolistsIncrementalIndexFactory.class);

	@Override
	public String type() {
		return ITodoUids.TYPE;
	}

	@Override
	public TypeIndexer create(BmContext ctx, Container c) {
		ITodoList todoListApi = ctx.provider().instance(ITodoList.class, c.uid);
		ElasticsearchClient esClient = ESearchActivator.getClient();
		VTodoIndexStore indexStore = new VTodoIndexStore(esClient, c, DataSourceRouter.location(ctx, c.uid));
		return new TodoIndexer(todoListApi, new ITodoListIndexWriter(indexStore));
	}

	public static class ITodoListIndexWriter implements IndexWriter<VTodo> {
		private VTodoIndexStore todoListIndexStore;

		public ITodoListIndexWriter(VTodoIndexStore cidx) {
			this.todoListIndexStore = cidx;
		}

		@Override
		public void write(ItemValue<VTodo> toIndex) {
			todoListIndexStore.update(toIndex.item(), toIndex.value);

		}

		@Override
		public void deleteByIds(List<Long> itemIds) {
			itemIds.forEach(item -> todoListIndexStore.delete(item));

		}

	}

	public static class TodoIndexer extends AbstractTypeIndexer<VTodo> {

		public TodoIndexer(ITodoList rec, IndexWriter<VTodo> writer) {
			super(rec, rec, writer);
		}

	}

}
