import { MailboxItem, MessageBody } from "@bluemind/backend.mail.api";
import { ItemValue } from "@bluemind/core.container.api";
import { extensions } from "@bluemind/extensions";
import { fetchRequest, dispatchFetch } from "@bluemind/service-worker-utils";
import session from "@bluemind/session";
import { exceptions, constants, helper, enums } from "@bluemind/smime.commons";

import pkcs7 from "../pkcs7";
import { getMyDecryptionKeyPair, getAllDecryptionKeyPairs } from "../pki";

import getImplementation from "./helpers/getSMimeImplementation";
import saveParts from "./saveParts";

const { ENCRYPTED_HEADER_NAME } = constants;
const { SmimeErrors } = exceptions;
const { CRYPTO_HEADERS } = enums;
const { addHeaderValue, resetHeader } = helper;

export type DecryptResult = {
    body: MessageBody;
    content: string;
};

export default async function (folderUid: string, item: ItemValue<MailboxItem>): Promise<DecryptResult> {
    let content = "";
    const { imapUid, body } = item.value;
    try {
        body.headers = resetHeader(body.headers, ENCRYPTED_HEADER_NAME);
        const { address, mime, encoding, charset, fileName } = body.structure!;

        const request = fetchRequest(session.sid, folderUid, imapUid!, address!, encoding!, mime!, charset!, fileName);
        const response = await dispatchFetch(request);
        const data = await response.blob();
        content = await doDecrypt(data, new Date(body.date!));
        const { structure, preview } = await saveParts(content, folderUid, item.value.imapUid!);
        body.preview = preview;
        body.structure = structure;
        body.headers = addHeaderValue(body?.headers, ENCRYPTED_HEADER_NAME, CRYPTO_HEADERS.OK);
    } catch (error: unknown) {
        const errorCode = error instanceof SmimeErrors ? error.code : CRYPTO_HEADERS.UNKNOWN;
        body.headers = addHeaderValue(body.headers, ENCRYPTED_HEADER_NAME, errorCode);
    }
    return { body, content };
}

async function doDecrypt(data: Blob, date: Date): Promise<string> {
    const implementation = getImplementation("decryptMimeEntity");
    return implementation(data, date);
}

extensions.register("smime.implementations", "smime.service-worker", {
    decryptMimeEntity: {
        fn: async function (data: Blob, date: Date) {
            const cachedPair = await getMyDecryptionKeyPair(date);
            try {
                return await pkcs7.decrypt(data, cachedPair.key, cachedPair.certificate);
            } catch (primaryError) {
                return retryWithFallbacks(data, date, primaryError);
            }
        },
        priority: 0
    }
});

async function retryWithFallbacks(data: Blob, date: Date, originalError: unknown): Promise<string> {
    const pairs = await getAllDecryptionKeyPairs(date);
    for (const pair of pairs) {
        try {
            const content = await pkcs7.decrypt(data, pair.key, pair.certificate);
            return content;
        } catch {
            continue;
        }
    }

    throw originalError;
}
