import { MessageBody } from "@bluemind/backend.mail.api";
import { helper } from "@bluemind/smime.commons";

import extractLeafs from "./extractLeafs";
import db from "./SMimeBodyDB";
import { checkParts } from "./SMimePartCache";

const { hasEncryptionHeader, hasSignatureHeader, isDecrypted, isVerified } = helper;

export async function getBody(guid: string, mappingFunction: () => Promise<MessageBody>): Promise<MessageBody> {
    let body = await db.getBody(guid);
    const isPartCacheValid = await hasValidCache(guid, body!);
    if (!isPartCacheValid) {
        body = await mappingFunction();
        if (isSuccess(body)) {
            await db.setBody(guid, body);
        }
    }
    return body!;
}
export function setReference(folderUid: string, imapUid: number, guid: string): Promise<void> {
    return db.setGuid(folderUid, imapUid, guid);
}
export function invalidate(): Promise<void> {
    const today = new Date();
    const timestamp = new Date(today.setDate(today.getDate() - 7)).getTime();
    return db.invalidate(timestamp);
}

function isSuccess(body: MessageBody): boolean {
    return (
        hasSignatureHeader(body.headers) === isVerified(body.headers) &&
        hasEncryptionHeader(body.headers) === isDecrypted(body.headers)
    );
}

async function hasValidCache(guid: MessageBody["guid"], cachedBody?: MessageBody): Promise<boolean> {
    if (!cachedBody) {
        return false;
    }

    const hasCacheableParts = isDecrypted(cachedBody.headers) || hasSignatureHeader(cachedBody.headers);
    if (!hasCacheableParts) {
        return true;
    }
    try {
        const parts = extractLeafs(cachedBody.structure!);
        return await checkParts(guid, parts);
    } catch {
        return false;
    }
}
