import { pki } from "node-forge";

import { MimeType } from "@bluemind/email";
import { extensions } from "@bluemind/extensions";
import BmRoles from "@bluemind/roles";

import { constants, enums } from "@bluemind/smime.commons";

import { clear as clearPki, getMyStatus, setMyPrivateKeys, setMyCertificates } from "./pki";
import { checkRecipientEmail, isPemReadable as pemToCertificate } from "./pki/checkCertificate";
import { isKeyReadable } from "./pki/checkKey";
import bodyDB from "./smime/cache/SMimeBodyDB";
import SMimeApiProxy from "./SMimeApiProxy";

const { SMIME_INTERNAL_API_URL, SMIME_UNTRUSTED_CERTIFICATE_ERROR_PREFIX } = constants;
const { PKIEntry } = enums;

extensions.register("serviceworker.handlers", "smime-plugin", {
    "api-handler": { class: SMimeApiProxy, priority: 256, role: BmRoles.CAN_USE_SMIME }
});

extensions.register("serviceworker.handlers", "smime-plugin", {
    "route-handler": {
        route: {
            capture: ({ url: { pathname } }) => pathname === SMIME_INTERNAL_API_URL,
            handler: hasCryptoFilesHandler,
            method: "GET"
        },
        priority: 128
    }
});
extensions.register("serviceworker.handlers", "smime-plugin", {
    "route-handler": {
        route: {
            capture: ({ url: { pathname } }) => pathname === SMIME_INTERNAL_API_URL,
            handler: clearPkiAndCache,
            method: "DELETE"
        },
        priority: 128
    }
});
extensions.register("serviceworker.handlers", "smime-plugin", {
    "route-handler": {
        route: {
            capture: ({ url: { pathname } }) => pathname === `${SMIME_INTERNAL_API_URL}/${PKIEntry.PRIVATE_KEYS}`,
            handler: setPrivateKeys,
            method: "PUT"
        },
        priority: 128
    }
});
extensions.register("serviceworker.handlers", "smime-plugin", {
    "route-handler": {
        route: {
            capture: ({ url: { pathname } }) => pathname === `${SMIME_INTERNAL_API_URL}/${PKIEntry.CERTIFICATES}`,
            handler: setCertificates,
            method: "PUT"
        },
        priority: 128
    }
});

async function hasCryptoFilesHandler() {
    return new Response(await getMyStatus());
}

async function clearPkiAndCache() {
    await clearPki();
    await bodyDB.clearBody();
    await bodyDB.clearGuid();
    await caches.delete("smime-part-cache");

    return new Response();
}

async function setPrivateKeys({ request }) {
    const blob = await request.blob();
    const pemKeys = await blob.text();
    try {
        const keys = JSON.parse(pemKeys).filter(isKeyReadable);
        await setMyPrivateKeys(new Blob([JSON.stringify(keys)], { type: MimeType.JSON }));
        return new Response();
    } catch (error) {
        return new Response(JSON.stringify({ message: error.message }), { status: 500 });
    }
}

async function setCertificates({ request }) {
    const expectedAddress = new URL(request.url).searchParams.get("email");
    const blob = await request.blob();
    const certificatesText = await blob.text();
    let certificates;
    try {
        certificates = JSON.parse(certificatesText).map(pemToCertificate);
    } catch (err) {
        return new Response(JSON.stringify({ message: err.message }), { status: 500 });
    }

    let error;
    const checkedCertificates = certificates
        .filter(certificate => {
            try {
                checkRecipientEmail(certificate, expectedAddress);
                return true;
            } catch (err) {
                error = err;
                return false;
            }
        })
        .map(pki.certificateToPem);
    if (checkedCertificates.length === 0 && error) {
        const message = `[${SMIME_UNTRUSTED_CERTIFICATE_ERROR_PREFIX}:${error.code}]` + error.message;
        return new Response(JSON.stringify({ message }), { status: 500 });
    }

    await setMyCertificates(new Blob([JSON.stringify(checkedCertificates)], { type: MimeType.JSON }));
    return new Response();
}
