import { DBSchema, IDBPDatabase, openDB } from "idb";

import logger from "@bluemind/logger";
import session from "@bluemind/session";
import { RevocationResult } from "@bluemind/smime.cacerts.api";
import { enums } from "@bluemind/smime.commons";

const { PKIEntry } = enums;

type RevocationSchema = RevocationResult & {
    cacheValidity: Date;
};

interface SMimePkiSchema extends DBSchema {
    my_keys_and_certs: {
        key: string;
        value: Blob;
    };
    revocations: {
        key: string;
        value: RevocationSchema;
    };
}

interface SMimePkiDB {
    clearMyCertsAndKeys(): Promise<void>;
    getPrivateKeys(): Promise<Blob | undefined>;
    setPrivateKeys(privateKey: Blob): Promise<void>;
    setCertificates(certificate: Blob): Promise<void>;
    getCertificates(): Promise<Blob | undefined>;
    getRevocation(serialNumber: string, issuerHash: string): Promise<RevocationSchema | undefined>;
    setRevocation(revocation: RevocationSchema, issuerHash: string): Promise<void>;
    clearRevocations(): Promise<void>;
}

class SMimePkiDBImpl implements SMimePkiDB {
    NAME = "smime:pki";
    VERSION = 2;
    connection: Promise<IDBPDatabase<SMimePkiSchema>>;
    constructor(userId: string) {
        this.connection = this.open(userId);
    }
    private async open(userId: string): Promise<IDBPDatabase<SMimePkiSchema>> {
        return openDB<SMimePkiSchema>(`${userId}:${this.NAME}`, this.VERSION, {
            upgrade: (db, from) => {
                logger.log(`[SW][SMimePkiDB] Upgrading from ${from} to ${this.VERSION}`);
                if (from < this.VERSION) {
                    logger.log("[SW][SMimePkiDB] Deleting existing object stores");
                    for (const name of Object.values(db.objectStoreNames)) {
                        db.deleteObjectStore(name);
                    }
                }
                db.createObjectStore("my_keys_and_certs");
                db.createObjectStore("revocations");
            },
            blocking: async () => {
                await this.close();
                this.connection = this.open(userId);
            }
        });
    }
    async close(): Promise<void> {
        (await this.connection).close();
    }
    async clearMyCertsAndKeys(): Promise<void> {
        return (await this.connection).clear("my_keys_and_certs");
    }
    async getPrivateKeys(): Promise<Blob | undefined> {
        return (await this.connection).get("my_keys_and_certs", PKIEntry.PRIVATE_KEYS);
    }
    async setPrivateKeys(privateKeys: Blob): Promise<void> {
        (await this.connection).put("my_keys_and_certs", privateKeys, PKIEntry.PRIVATE_KEYS);
    }
    async setCertificates(certificates: Blob): Promise<void> {
        (await this.connection).put("my_keys_and_certs", certificates, PKIEntry.CERTIFICATES);
    }
    async getCertificates(): Promise<Blob | undefined> {
        return (await this.connection).get("my_keys_and_certs", PKIEntry.CERTIFICATES);
    }
    async getRevocation(serialNumber: string, issuerHash: string): Promise<RevocationSchema | undefined> {
        return (await this.connection).get("revocations", issuerHash + "-" + serialNumber);
    }
    async setRevocation(revocation: RevocationSchema, issuerHash: string): Promise<void> {
        (await this.connection).put("revocations", revocation, issuerHash + "-" + revocation.revocation.serialNumber);
    }
    async clearRevocations(): Promise<void> {
        return (await this.connection).clear("revocations");
    }
}

let implementation: SMimePkiDBImpl | null = null;
async function instance(): Promise<SMimePkiDB> {
    if (!implementation) {
        implementation = new SMimePkiDBImpl(session.userId);
    }
    return implementation;
}

session.addEventListener("change", event => {
    const { old, value } = event.detail;
    if (value.userId != old?.userId && implementation) {
        implementation?.close();
        implementation = null;
    }
});

const db: SMimePkiDB = {
    clearMyCertsAndKeys: () => instance().then(db => db.clearMyCertsAndKeys()),
    getPrivateKeys: () => instance().then(db => db.getPrivateKeys()),
    setPrivateKeys: setPrivateKeys => instance().then(db => db.setPrivateKeys(setPrivateKeys)),
    setCertificates: certificate => instance().then(db => db.setCertificates(certificate)),
    getCertificates: () => instance().then(db => db.getCertificates()),
    getRevocation: (serialNumber, issuerHash) => instance().then(db => db.getRevocation(serialNumber, issuerHash)),
    setRevocation: (revocation, issuerHash) => instance().then(db => db.setRevocation(revocation, issuerHash)),
    clearRevocations: () => instance().then(db => db.clearRevocations())
};

export default db;
