/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.memory.pool.api;

import java.io.IOException;
import java.util.function.Supplier;

import com.google.common.base.Suppliers;
import com.netflix.spectator.api.Gauge;
import com.netflix.spectator.api.Registry;

import net.bluemind.memory.pool.api.MemoryMappedPool.PoolStats;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;

public class CommonMemoryPool {

	private CommonMemoryPool() {
	}

	private static final Supplier<MemoryMappedPool> POOL = Suppliers.memoize(() -> {
		MemoryMappedPool ret;
		try {
			ret = Pools.factory().create();
		} catch (IOException e) {
			throw new UnsupportedOperationException(e);
		}

		setupMetrics(ret);

		Runtime.getRuntime().addShutdownHook(new Thread(() -> {
			try {
				ret.close();
			} catch (Exception e) {
				// ok
			}
		}));
		return ret;
	});

	public static MemoryMappedPool getDefault() {
		return POOL.get();
	}

	private static void setupMetrics(MemoryMappedPool ret) {
		Registry reg = MetricsRegistry.get();
		String jvm = System.getProperty("net.bluemind.property.product", "unknown");
		IdFactory idf = new IdFactory("memory-pool", reg, CommonMemoryPool.class);
		PoolStats initialStats = ret.getStats();
		Gauge activeSegs = reg.gauge(idf.name("active-segments", "jvm", jvm));
		activeSegs.set(initialStats.getActiveSegments());
		Gauge usedSize = reg.gauge(idf.name("used-size", "jvm", jvm));
		usedSize.set(initialStats.getUsedSize());

		Thread.ofVirtual().name("pool-metrics").start(() -> {
			while (true) {
				try {
					Thread.sleep(10000);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				}
				PoolStats liveStats = ret.getStats();
				activeSegs.set(liveStats.getActiveSegments());
				usedSize.set(liveStats.getUsedSize());
			}
		});
	}

}
