/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.memory.pool.api;

import java.io.IOException;
import java.lang.ref.Cleaner;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkFiler {

	private static final Cleaner chunkRefsCleander = Cleaner.create(Thread.ofVirtual().factory());
	private static final ConcurrentHashMap<UUID, IWritableChunk> GLOBAL_INDEX = new ConcurrentHashMap<>();
	private static final Logger logger = LoggerFactory.getLogger(ChunkFiler.class);

	static {
		Thread.ofVirtual().name("drop-released").start(() -> {
			while (true) {
				try {
					Thread.sleep(5000);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				}
				GLOBAL_INDEX.entrySet().removeIf(e -> e.getValue().isReleased());
			}
		});
	}

	private ChunkFiler() {
	}

	public static final String PREFIX = "chunk://";
	public static final int PF_LEN = PREFIX.length();

	public static record NamedChunk(String name, IWritableChunk chunk) {

	}

	private static record Cleanup(UUID k) implements Runnable {

		@Override
		public void run() {
			IWritableChunk chunk = GLOBAL_INDEX.remove(k);
			chunk.release();
			logger.debug("Released {} for {}, allocation stack follows", chunk, k);
		}

	}

	public static NamedChunk newNamedChunk(int size) throws IOException {
		UUID random = UUID.randomUUID();
		int validSize = Math.max(size, 4096);

		IWritableChunk alloc = CommonMemoryPool.getDefault().allocateEmpty(validSize);
		GLOBAL_INDEX.put(random, alloc);
		logger.debug("Alloc {} -> {}", random, alloc);
		String addr = PREFIX + random.toString();

		chunkRefsCleander.register(addr, new Cleanup(random));
		return new NamedChunk(addr, alloc);
	}

	public static String newChunkAddress(int size) throws IOException {
		return newNamedChunk(size).name();
	}

	public static IWritableChunk byAddress(String address) {
		UUID key = UUID.fromString(address.substring(PF_LEN));
		IWritableChunk ret = GLOBAL_INDEX.get(key);
		if (ret == null) {
			throw new NullPointerException("No chunk at address " + address);
		}
		return ret;
	}

}
