/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.utils.sessions;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AsyncResult;
import io.vertx.core.MultiMap;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpClientResponse;
import io.vertx.core.http.HttpHeaders;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.RequestOptions;
import io.vertx.core.json.JsonObject;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.utils.endpoint.KeycloakEndpoints;

public class OpenIdSessionRefresher {
	private static final Logger logger = LoggerFactory.getLogger(OpenIdSessionRefresher.class);

	private final String realm;
	private final String sessionId;
	private final String openIdClientSecret;
	private final JsonObject jwt;
	private final Consumer<JsonObject> refreshedJwt;

	public OpenIdSessionRefresher(String realm, String sessionId, String openIdClientSecret, JsonObject jwt,
			Consumer<JsonObject> refreshedJwt) {
		this.realm = realm;
		this.sessionId = sessionId;
		this.openIdClientSecret = openIdClientSecret;
		this.jwt = jwt;
		this.refreshedJwt = refreshedJwt;
	}

	public void refresh(HttpClient httpClient) {
		if (logger.isDebugEnabled()) {
			logger.debug("Refresh session {}, using endpoint: {}", sessionId, KeycloakEndpoints.tokenEndpoint(realm));
		}

		httpClient.request(
				new RequestOptions().setMethod(HttpMethod.POST).setAbsoluteURI(KeycloakEndpoints.tokenEndpoint(realm)),
				this::decorateRequest);
	}

	private void decorateRequest(AsyncResult<HttpClientRequest> request) {
		if (!request.succeeded()) {
			logger.error("Unable to refresh session {}", sessionId, request.cause());
			return;
		}

		HttpClientRequest r = request.result();
		r.response(this::refreshResponse);

		if (logger.isDebugEnabled()) {
			String[] chunks = jwt.getString("refresh_token").split("\\.");
			logger.debug("Refresh token: chunk0: {}, chunk1: {}", new String(Base64.getUrlDecoder().decode(chunks[0])),
					new String(Base64.getUrlDecoder().decode(chunks[1])));
		}

		byte[] postData = new StringBuilder().append("client_id=").append(IKeycloakUids.clientId(realm)) //
				.append("&client_secret=").append(openIdClientSecret).append("&grant_type=refresh_token&refresh_token=")
				.append(jwt.getString("refresh_token")).toString().getBytes(StandardCharsets.UTF_8);

		MultiMap headers = r.headers();
		headers.add(HttpHeaders.ACCEPT_CHARSET, StandardCharsets.UTF_8.name());
		headers.add(HttpHeaders.CONTENT_TYPE, "application/x-www-form-urlencoded");
		headers.add(HttpHeaders.CONTENT_LENGTH, Integer.toString(postData.length));
		r.write(Buffer.buffer(postData));
		r.end();
	}

	private void refreshResponse(AsyncResult<HttpClientResponse> response) {
		if (!response.succeeded()) {
			logger.error("Fail to refresh session {}", sessionId, response.cause());
			return;
		}

		response.result().body(body -> {
			if (response.result().statusCode() != 200) {
				logger.error("Fail to refresh session {}, response status {}, content {}", sessionId,
						response.result().statusCode(), new String(body.result().getBytes()));
				return;
			}

			try {
				refreshedJwt.accept(new JsonObject(new String(body.result().getBytes())));
			} catch (Exception e) {
				logger.warn("Unable to decode session {} refresh response: {}", sessionId, e.getMessage());
				return;
			}

			if (logger.isDebugEnabled()) {
				logger.debug("Session {} refreshed successfully", sessionId);
			}
		});
	}
}
