/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.utils;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Base64;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.auth.AuthDomainProperties;
import net.bluemind.core.api.auth.AuthTypes;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.DomainSettingsKeys;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.keycloak.api.IKeycloakKerberosAdmin;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.utils.adapters.KerberosComponentAdapter;
import net.bluemind.network.topology.Topology;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.server.api.TagDescriptor;

public class KerberosConfigHelper {
	private static final Logger logger = LoggerFactory.getLogger(KerberosConfigHelper.class);
	private static final String LAST_CONF_LOCATION = "/etc/bm-keycloak/krbconf.json";
	private static final String GLOBAL_VIRT = "global.virt";

	private KerberosConfigHelper() {

	}

	public static void updateKeycloakKerberosConf(ItemValue<Domain> domain) {
		String domainUid = domain.uid;
		if (GLOBAL_VIRT.equals(domainUid)) {
			return;
		}
		if (domain.value.properties == null
				|| domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()) == null) {
			logger.warn("skipping kerberos conf update for domain {} (no domain properties)", domainUid);
			return;
		}

		logger.info("Domain {} created/updated : updating kerberos conf", domainUid);

		ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IKeycloakKerberosAdmin.class, domainUid)
				.deleteKerberosProvider(IKeycloakUids.kerberosComponentName(IKeycloakUids.realmId(domainUid)));

		if (AuthTypes.KERBEROS.name().equals(domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()))) {
			createKeycloakKerberosConf(domain);
		}
		KerberosConfigHelper.updateGlobalRealmKerb();
		KerberosConfigHelper.updateKrb5Conf();
	}

	public static void createKeycloakKerberosConf(ItemValue<Domain> domain) {
		String krbKeytab = domain.value.properties.get(AuthDomainProperties.KRB_KEYTAB.name());

		Map<String, String> domainSettings = Shared.mapDomainSettings().get(domain.uid);
		String domainExternalUrl = domainSettings.get(DomainSettingsKeys.external_url.name());

		String keytabPath = getKeytabFilename(domain.uid);
		String kcServerAddr = Topology.get().any(TagDescriptor.bm_keycloak.getTag()).value.address();
		INodeClient nodeClient = NodeActivator.get(kcServerAddr);
		nodeClient.writeFile(keytabPath, new ByteArrayInputStream(Base64.getDecoder().decode(krbKeytab)));

		KerberosComponentAdapter kca = KerberosComponentAdapter.build(domain);

		if (!GLOBAL_VIRT.equals(domain.uid) && domainExternalUrl == null) {
			IKeycloakKerberosAdmin kerbProv = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IKeycloakKerberosAdmin.class, GLOBAL_VIRT);
			kerbProv.deleteKerberosProvider(IKeycloakUids.kerberosComponentName(GLOBAL_VIRT));
			kerbProv.create(kca.component);
		} else {
			ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
					.instance(IKeycloakKerberosAdmin.class, domain.uid).create(kca.component);
		}
	}

	public static void updateGlobalRealmKerb() {
		ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IInCoreDomains domainService = provider.instance(IInCoreDomains.class);

		if (domainService.allUnfiltered().stream().noneMatch(domain -> isKerberosWithoutExternalUrl(domain))) {
			IKeycloakKerberosAdmin kerbProv = provider.instance(IKeycloakKerberosAdmin.class, GLOBAL_VIRT);
			kerbProv.deleteKerberosProvider(IKeycloakUids.kerberosComponentName(GLOBAL_VIRT));
		}
	}

	private static boolean isKerberosWithoutExternalUrl(ItemValue<Domain> domain) {
		return AuthTypes.KERBEROS.name().equals(domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()))
				&& getExternalUrl(domain.uid) == null;
	}

	public static void updateKrb5Conf() {
		JsonObject currentConf = getConf();
		JsonObject previousConf = null;
		try {
			previousConf = new JsonObject(Files.readString(Paths.get(LAST_CONF_LOCATION)));
		} catch (IOException e) {
			previousConf = new JsonObject();
		}

		if (!currentConf.equals(previousConf)) {
			String kcServerAddr = Topology.get().any(TagDescriptor.bm_keycloak.getTag()).value.address();
			INodeClient nodeClient = NodeActivator.get(kcServerAddr);

			nodeClient.writeFile(LAST_CONF_LOCATION,
					new ByteArrayInputStream(currentConf.encode().getBytes(StandardCharsets.UTF_8)));

			nodeClient.listFiles("/etc/bm-keycloak/", "keytab").forEach(file -> nodeClient.deleteFile(file.getPath()));
			currentConf.fieldNames().forEach(domainUid -> nodeClient.writeFile(getKeytabFilename(domainUid),
					new ByteArrayInputStream(Base64.getDecoder().decode(
							currentConf.getJsonObject(domainUid).getString(AuthDomainProperties.KRB_KEYTAB.name())))));
		} else {
			logger.debug("Kerberos config did not change. No need to update /etc/krb5.conf.");
		}
	}

	private static JsonObject getConf() {
		JsonObject conf = new JsonObject();

		ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IInCoreDomains.class).allUnfiltered()
				.forEach(domain -> {
					if (AuthTypes.KERBEROS.name()
							.equals(domain.value.properties.get(AuthDomainProperties.AUTH_TYPE.name()))) {
						conf.put(domain.uid,
								new JsonObject()
										.put(AuthDomainProperties.KRB_AD_DOMAIN.name(),
												domain.value.properties.get(AuthDomainProperties.KRB_AD_DOMAIN.name()))
										.put(AuthDomainProperties.KRB_AD_IP.name(),
												domain.value.properties.get(AuthDomainProperties.KRB_AD_IP.name()))
										.put(AuthDomainProperties.KRB_KEYTAB.name(),
												domain.value.properties.get(AuthDomainProperties.KRB_KEYTAB.name())));
					}
				});

		return conf;
	}

	private static String getExternalUrl(String domainUid) {
		return Shared.mapDomainSettings().get(domainUid).get(DomainSettingsKeys.external_url.name());
	}

	public static String getKeytabFilename(String domainUid) {
		return "/etc/bm-keycloak/" + domainUid + ".keytab";
	}

}
