/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.imap.driver.mailapi;

import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.IntStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import co.elastic.clients.elasticsearch._types.query_dsl.BoolQuery;
import co.elastic.clients.elasticsearch._types.query_dsl.QueryBuilders;
import net.bluemind.imap.driver.mailapi.UidSearchAnalyzer.QueryBuilderResult;

public class UidSearchWellKnownPatternForSlowQuery {

	public interface IFastPathForSlowQuery {
		public boolean match(String query);

		public QueryBuilderResult computeQuery(String query, String folderUid);

	}

	@SuppressWarnings("serial")
	public static class CannotParseFastPathException extends RuntimeException {
		public CannotParseFastPathException(String message) {
			super(message);
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(UidSearchWellKnownPatternForSlowQuery.class);
	private static List<IFastPathForSlowQuery> specialQueries = List.of(new UndeletedWithSeveralOrPattern(),
			new NotdeletedWithSeveralOrPattern(), new UnDeletedOrWithMessageId());

	private static final String PARENT_TYPE = "body";

	public static Optional<IFastPathForSlowQuery> lookup(String query) {
		return specialQueries.stream().filter(sp -> sp.match(query)).findAny().map(fp -> {
			logger.info("Using fast-path {} for '{}'", fp, query);
			return fp;
		});
	}

	/***
	 * fast path for search queries like : UNDELETED (OR (OR (OR (OR (OR (OR FROM
	 * "Alice" OR TO "Alice" HEADER CC "Alice") SUBJECT "Alice") BODY "Alice") FROM
	 * "BlueMind" OR TO "BlueMind" HEADER CC "BlueMind") SUBJECT "BlueMind") BODY
	 * "BlueMind")"
	 */
	private static class UndeletedWithSeveralOrPattern implements IFastPathForSlowQuery {
		private static String UNDELETED_PATTERN = "(?:from \"(\\w+)\" or to \"(\\w+)\" header cc \"(\\w+)\"\\) subject \"(\\w+)\"\\) body \"(\\w+)\"\\))+";

		@Override
		public boolean match(String query) {

			boolean startMatch = Pattern.compile("^undeleted (?:\\(or\\s)+", Pattern.CASE_INSENSITIVE).matcher(query)
					.find();
			if (!startMatch) {
				return false;
			}

			boolean queryMatch = Pattern.compile(UNDELETED_PATTERN, Pattern.CASE_INSENSITIVE).matcher(query).find();
			return queryMatch;
		}

		@Override
		public QueryBuilderResult computeQuery(String query, String folderUid) {

			Matcher matcher = Pattern.compile(UNDELETED_PATTERN, Pattern.CASE_INSENSITIVE).matcher(query);
			try {
				return buildNotDeletedSeveralOrQueryBuilderResult(matcher, folderUid);
			} catch (CannotParseFastPathException e) {
				logger.warn("Cannot find fastpath for query '{}'", query);
				return null;
			}
		}

	}

	/***
	 * fast path for search queries like : NOT DELETED OR OR OR OR OR FROM "Bob" TO
	 * "Bob" CC "Bob" BCC "Bob" BODY "Bob" SUBJECT "Bob" OR OR OR OR OR FROM
	 * "BlueMind" TO "BlueMind" CC "BlueMind" BCC "BlueMind" BODY "BlueMind" SUBJECT
	 * "BlueMind"
	 */
	private static class NotdeletedWithSeveralOrPattern implements IFastPathForSlowQuery {
		private static String NOT_DELETED_PATTERN = "(?:from \"(\\w+)\" to \"(\\w+)\" cc \"(\\w+)\" bcc \"(\\w+)\" body \"(\\w+)\" subject \"(\\w+)\")+";

		@Override
		public boolean match(String query) {

			boolean startMatch = Pattern.compile("^not deleted (?:or\\s)+", Pattern.CASE_INSENSITIVE).matcher(query)
					.find();
			if (!startMatch) {
				return false;
			}

			boolean queryMatch = Pattern.compile(NOT_DELETED_PATTERN, Pattern.CASE_INSENSITIVE).matcher(query).find();
			return queryMatch;
		}

		@Override
		public QueryBuilderResult computeQuery(String query, String folderUid) {
			Matcher matcher = Pattern.compile(NOT_DELETED_PATTERN, Pattern.CASE_INSENSITIVE).matcher(query);
			// BCC information is not stored in ES -> do not take it into account
			try {
				return buildNotDeletedSeveralOrQueryBuilderResult(matcher, folderUid);
			} catch (CannotParseFastPathException e) {
				logger.warn("Cannot find fastpath for query '{}'", query);
				return null;
			}
		}

	}

	/***
	 * fast path for search queries like : OR HEADER Message-ID
	 * <TDGX04W007CS6002AWY4W1727951421525@test.com> OR HEADER Message-ID
	 * <48ac72ac87cc9ab62e26c8ba3e6fe250@test.com> OR HEADER Message-ID
	 * <66FE627B.4070901@test.com> OR HEADER Message-ID
	 * <410f1082f8f34c448dbb253df427061a@test.com> HEADER Message-ID
	 * <a1d920e6123f36b56e97279b6d694339@test.com> UNDELETED
	 */
	private static class UnDeletedOrWithMessageId implements IFastPathForSlowQuery {
		private static String MESSAGE_ID_PATTERN = "HEADER Message-ID (\\S+)\\s";

		@Override
		public boolean match(String query) {

			boolean startMatch = Pattern.compile("^or header message-id .+ UNDELETED$", Pattern.CASE_INSENSITIVE)
					.matcher(query).find();
			if (!startMatch) {
				return false;
			}

			boolean queryMatch = Pattern.compile(MESSAGE_ID_PATTERN, Pattern.CASE_INSENSITIVE).matcher(query).find();
			return queryMatch;
		}

		@Override
		public QueryBuilderResult computeQuery(String query, String folderUid) {
			Matcher matcher = Pattern.compile(MESSAGE_ID_PATTERN, Pattern.CASE_INSENSITIVE).matcher(query);
			BoolQuery.Builder qb = new BoolQuery.Builder() //
					.must(m -> m.term(t -> t.field("in").value(folderUid))) //
					.mustNot(m -> m.term(t -> t.field("is").value("deleted")));
			BoolQuery.Builder qbShould = new BoolQuery.Builder();
			matcher.results().filter(r -> r.groupCount() > 0).map(r -> r.group(1))
					.forEach(messageId -> qbShould.should(m -> m.hasParent(p -> p.parentType(PARENT_TYPE).score(false) //
							.query(q -> q.term(ph -> ph.field("messageId").value(messageId))))));
			qb.must(qbShould.build()._toQuery());
			return new QueryBuilderResult(qb.build()._toQuery(), false);
		}

	}

	private static QueryBuilderResult buildNotDeletedSeveralOrQueryBuilderResult(Matcher matcher, String folderUid)
			throws CannotParseFastPathException {

		BoolQuery.Builder qb = new BoolQuery.Builder() //
				.must(m -> m.term(t -> t.field("in").value(folderUid)));
		qb.mustNot(m -> m.term(t -> t.field("is").value("deleted")));

		// Check associated values to keywords are the same into the pattern
		matcher.results().forEach(result -> {
			if (result.groupCount() > 0) {
				var list = IntStream.range(1, result.groupCount()).mapToObj(i -> result.group(i)).distinct().toList();
				if (list.size() > 1) {
					throw new CannotParseFastPathException("query is not correctly formatted for fast parsing");
				}
			}
		});
		matcher.reset();
		String fullQuery = matcher.results().map(r -> r.group(1)).reduce((p, e) -> p + " " + e).orElse(null);
		if (fullQuery == null) {
			throw new CannotParseFastPathException("query is not correctly formatted for fast parsing");
		}
		BoolQuery.Builder qbShould = new BoolQuery.Builder();
		qbShould.should(List.of(QueryBuilders.hasParent(p -> p.parentType(PARENT_TYPE).score(false) //
				.query(q -> q.matchPhrase(ph -> ph.field("from").query(fullQuery)))),
				QueryBuilders.hasParent(p -> p.parentType(PARENT_TYPE).score(false) //
						.query(q -> q.matchPhrase(ph -> ph.field("to").query(fullQuery)))),
				QueryBuilders.hasParent(p -> p.parentType(PARENT_TYPE).score(false) //
						.query(q -> q.matchPhrase(ph -> ph.field("subject").query(fullQuery)))),
				QueryBuilders.hasParent(p -> p.parentType(PARENT_TYPE).score(false) //
						.query(q -> q.matchPhrase(ph -> ph.field("content").query(fullQuery)))),
				QueryBuilders.hasParent(p -> p.parentType(PARENT_TYPE).score(false) //
						.query(q -> q.matchPhrase(ph -> ph.field("headers.cc").query(fullQuery))))));
		qb.must(qbShould.build()._toQuery());
		return new QueryBuilderResult(qb.build()._toQuery(), false);
	}

}
