/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.backup.continuous.restore.orphans;

import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.core.type.TypeReference;

import net.bluemind.core.backup.continuous.DataElement;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.eas.api.Account;
import net.bluemind.eas.api.FolderSyncVersions;
import net.bluemind.eas.api.Heartbeat;
import net.bluemind.eas.persistence.EasStore;

public class RestoreEasSync {

	private static final ValueReader<ItemValue<FolderSyncVersions>> folderSyncReader = JsonUtils
			.reader(new TypeReference<ItemValue<FolderSyncVersions>>() {
			});
	private static final ValueReader<ItemValue<Heartbeat>> heartbeatReader = JsonUtils
			.reader(new TypeReference<ItemValue<Heartbeat>>() {
			});
	private static final ValueReader<ItemValue<Account>> resetReader = JsonUtils
			.reader(new TypeReference<ItemValue<Account>>() {
			});
	private static final ValueReader<ItemValue<String>> clientIdReader = JsonUtils
			.reader(new TypeReference<ItemValue<String>>() {
			});
	private IServerTaskMonitor monitor;
	private EasStore store;

	public RestoreEasSync(IServerTaskMonitor monitor) {
		this.monitor = monitor;
		BmContext ctx = ServerSideServiceProvider
				.getProvider(new SecurityContext("sid", "no", List.of(), List.of(), "global.virt")).getContext();
		this.store = new EasStore(ctx.getDataSource());
	}

	public void restoreFolderSync(DataElement deFoldersync) {
		try {
			ItemValue<FolderSyncVersions> iv = folderSyncReader.read(deFoldersync.payload);
			if (deFoldersync.isDelete()) {
				store.setFolderSyncVersions(iv.value.account, Collections.emptyMap());
			} else if (iv.value != null) {
				store.setFolderSyncVersions(iv.value.account, iv.value.versions);
			}
		} catch (Exception e) {
			monitor.log("error on restore EAS foldersync", e);
		}
	}

	public void restoreHeatbeat(DataElement heatbeat) {
		try {
			ItemValue<Heartbeat> iv = heartbeatReader.read(heatbeat.payload);
			if (!heatbeat.isDelete() && iv.value != null) {
				store.setHeartbeat(iv.value);
			}
		} catch (Exception e) {
			monitor.log("error on restore EAS heatbeat", e);
		}
	}

	public void restoreReset(DataElement deReset) {
		try {
			ItemValue<Account> iv = resetReader.read(deReset.payload);
			if (deReset.key.operation.equals("DELETE")) {
				store.deletePendingReset(iv.value);
			} else if (iv.value != null) {
				store.insertPendingReset(iv.value);
				monitor.log("Created EAS pending reset {}", iv.value);
			}
		} catch (Exception e) {
			monitor.log("error on restore EAS pending reset", e);
		}
	}

	public void restoreClientId(DataElement deClientId) {
		try {
			ItemValue<String> iv = clientIdReader.read(deClientId.payload);
			if (iv.value != null) {
				store.insertClientId(iv.value);
				monitor.log("Created EAS clientId {}", iv.value);
			}
		} catch (Exception e) {
			monitor.log("error on restore EAS clientId", e);
		}

	}

}
