package net.bluemind.core.backup.continuous.restore.domains.replication;

import java.util.concurrent.Semaphore;

import com.fasterxml.jackson.core.type.TypeReference;

import net.bluemind.backend.mail.replica.indexing.RecordIndexActivator;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.backup.continuous.dto.IndexedMessageBodyDTO;
import net.bluemind.core.backup.continuous.dto.VersionnedItem;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.model.RecordKey.Operation;
import net.bluemind.core.backup.continuous.restore.domains.RestoreDomainType;
import net.bluemind.core.backup.continuous.restore.domains.RestoreLogger;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;

public class RestoreMessageBodyESSource implements RestoreDomainType {

	private final RestoreLogger log;
	private final boolean onlyRestoreBodiesFromBodiesStore;
	private static final Semaphore concurrent = new Semaphore(128); // Maximum requests in parallel to elasticsearch

	public RestoreMessageBodyESSource(RestoreLogger log, boolean onlyRestoreBodiesFromBodiesStore) {
		this.log = log;
		this.onlyRestoreBodiesFromBodiesStore = onlyRestoreBodiesFromBodiesStore;

	}

	private static final ValueReader<VersionnedItem<IndexedMessageBodyDTO>> reader = JsonUtils
			.reader(new TypeReference<VersionnedItem<IndexedMessageBodyDTO>>() {
			});

	@Override
	public String type() {
		return "message_bodies_es_source";
	}

	@Override
	public void restore(RecordKey key, String payload) {
		if (onlyRestoreBodiesFromBodiesStore && !key.uid.contains("_at_bodies.store_")) {
			return;
		}
		create(key, payload);
	}

	private void create(RecordKey key, String payload) {
		if (Operation.isDelete(key)) {
			log.skip(type(), key, "Skip ES deletion, mailspool_pending will expire");
			return;
		}
		concurrent.acquireUninterruptibly();
		// Because this operation is very slow and IO dependent, we don't want to
		// block all other processing while this one is running
		Thread.ofVirtual().name("es-index-body-source").start(() -> {
			try {
				VersionnedItem<IndexedMessageBodyDTO> item = reader().read(payload);
				RecordIndexActivator.getIndexer().ifPresentOrElse(service -> {
					if (item.value.data != null) {
						service.storeBodyAsByte(item.uid, item.value.data);
						log.create(type(), key);
					}
				}, () -> {
					log.skip(type(), key, "ES source with " + item.value.data.length + " byte(s)");
					throw new RuntimeException("Record index activator is needed for clean forks");
				});
			} finally {
				concurrent.release();
			}
		});
	}

	private ValueReader<VersionnedItem<IndexedMessageBodyDTO>> reader() {
		return reader;
	}
}
