package net.bluemind.core.backup.continuous.restore.domains;

import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.core.type.TypeReference;

import io.vertx.core.Handler;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.backup.continuous.DataElement;
import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.model.RecordKey.Operation;
import net.bluemind.core.backup.continuous.restore.IOwnerChecker;
import net.bluemind.core.backup.continuous.restore.ISeppukuAckListener;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreCalendarView;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreDeferredAction;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreDevice;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreMailboxIdentity;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreMailflow;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreOwnerSubscriptions;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreResourceType;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreSmimeCacert;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreTags;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreUserAccounts;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreUserCredentials;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreUserMailIdentities;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreVCard;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreVEventSeries;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreVNote;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreVTodo;
import net.bluemind.core.backup.continuous.restore.domains.crud.RestoreWebAppData;
import net.bluemind.core.backup.continuous.restore.domains.crud.directories.RestoreDirectories;
import net.bluemind.core.backup.continuous.restore.domains.replication.RestoreMailboxRecords;
import net.bluemind.core.backup.continuous.restore.domains.replication.RestoreMessageBody;
import net.bluemind.core.backup.continuous.restore.domains.replication.RestoreMessageBodyESSource;
import net.bluemind.core.backup.continuous.restore.domains.replication.RestoreReplicatedMailboxes;
import net.bluemind.core.backup.continuous.tools.DeletionHelper;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.domain.api.Domain;

public class DomainRestorationHandler implements Handler<DataElement> {

	private final RestoreLogger log;
	private final Map<String, RestoreDomainType> restoresByType;
	private final Set<String> skip;
	private final RestoreState state;
	private final IOwnerChecker ownerChecker;
	private final IServiceProvider target;
	private final ItemValue<Domain> domain;

	public DomainRestorationHandler(IServerTaskMonitor monitor, Set<String> skip, ItemValue<Domain> domain,
			IServiceProvider target, ISeppukuAckListener byeAck, RestoreState state, IOwnerChecker ownerChecker,
			boolean onlyRestoreBodiesFromBodiesStore) {
		this.state = state;
		this.log = new RestoreLogger(monitor);
		this.skip = skip;
		this.ownerChecker = ownerChecker;
		this.target = target;
		this.domain = domain;
		this.restoresByType = Arrays.asList(//
				new RestoreMailboxRecords(log, state, domain, target), //
				new RestoreMessageBody(log, target, onlyRestoreBodiesFromBodiesStore), //
				new RestoreMessageBodyESSource(log, onlyRestoreBodiesFromBodiesStore), //
				new RestoreDirectories(log, domain, target, byeAck, state, ownerChecker), //
				new RestoreReplicatedMailboxes(log, domain, state, target), //
				new RestoreMapiArtifacts(log, domain, target), //
//				new RestoreMapiFAI(log, domain, target),
				new RestoreFlatHierarchy(log, domain, target, state), //
				new RestoreVCard(log, domain, target, state), //
				new RestoreVEventSeries(log, domain, target, state), //
				new RestoreDeferredAction(log, domain, target, state), //
				new RestoreVTodo(log, domain, target, state), //
				new RestoreVNote(log, domain, target, state), //
				new RestoreMembership(log, domain, target, state), //
				new RestoreRoles(log, domain, target, state), //
				new RestoreOrgUnitAdminRoles(log, domain, target), //
				new RestoreResourceType(log, domain, target, state), //
				new RestoreMailFilter(log, domain, target), //
				new RestoreContainerMetadata(log, target, state), //
				new RestoreOwnerSubscriptions(log, domain, target, state), //
				new RestoreTags(log, domain, target, state), //
				new RestoreDevice(log, domain, target, state), //
				new RestoreMailflow(log, domain, target, state), //
				new RestoreUserAccounts(log, domain, target, state), //
				new RestoreMailboxIdentity(log, domain, target, state), //
				new RestoreUserMailIdentities(log, domain, target, state), //
				new RestoreWebAppData(log, domain, target, state), //
				new RestoreCalendarView(log, domain, target, state), //
				new RestoreSmimeCacert(log, domain, target, state), //
				new RestoreUserCredentials(log, domain, target), //
				new RestorePcl(log, target) //
		).stream().collect(Collectors.toMap(RestoreDomainType::type, Function.identity()));
	}

	@JsonIgnoreProperties(ignoreUnknown = true)
	private static class WithUid {
		public String uid;
	}

	private static final ValueReader<WithUid> justUid = JsonUtils.reader(new TypeReference<WithUid>() {
	});

	private static final Set<String> HAS_VIRTUAL_OWNER = Set.of("message_bodies", "message_bodies_es_source");

	@Override
	public void handle(DataElement event) {
		fixupKey(event.key);
		boolean isOpDelete = Operation.isDelete(event.key);
		boolean endsUpDeteled = ownerChecker.isKnownDeletion(event.key);
		if (!isOpDelete && endsUpDeteled) {
			log.skip(event.key.type, event.key, "Index has deletion");
			return;
		} else if (isOpDelete && endsUpDeteled && event.payload != null) {
			touchVersionForDeletion(event);
		}

		RestoreDomainType restore = restoresByType.get(event.key.type);
		String payload = new String(event.payload);

		if (restore != null && !skip.contains(event.key.type)) {
			handle0(restore, event, payload);
		} else {
			log.skip(event.key.type, event.key, payload);
		}
	}

	private boolean touchVersionForDeletion(DataElement event) {
		// we skipped CREATE & DELETE but we want to get the version right
		JsonObject js = new JsonObject(new String(event.payload));
		if (js.containsKey("uid") && js.containsKey("version")) {
			ItemVersion iv = new ItemVersion(event.key.id, js.getLong("version"), new Date());
			IContainers contApi = target.instance(IContainers.class);
			BaseContainerDescriptor parentContainer = contApi.getLightIfPresent(event.key.uid);
			if (parentContainer != null) {
				DeletionHelper dh = DeletionHelper.forContainer(parentContainer);
				dh.restoreDelete(iv);
				return true;
			}
		}
		return false;
	}

	private void handle0(RestoreDomainType restore, DataElement event, String payload) {
		boolean ignoreFailure = false;

		if (!ownerChecker.isKnown(event.key.owner)) {
			if (!HAS_VIRTUAL_OWNER.contains(event.key.type)) {
				log.monitor().log("[{}:{}] owner of {} is not known to index: ignore errors", event.part, event.offset,
						event.key);
				ignoreFailure = true;
			}
		} else if (event.key.owner.equals("system") && event.key.type.equals("dir") && event.payload != null) {
			WithUid checked = justUid.read(event.payload);
			if (!ownerChecker.isKnown(checked.uid)) {
				log.monitor().log("[{}:{}] owner of {} is not known to index: ignore errors", event.part, event.offset,
						event.key);
				ignoreFailure = true;
			}
		} else if (!event.key.type.equals("dir") && ownerChecker.outOfOrderCreate(event.key.owner).isPresent()) {
			DataElement idxElem = ownerChecker.outOfOrderCreate(event.key.owner).orElseThrow();
			RestoreDomainType restoreOwner = restoresByType.get(idxElem.key.type);
			log.monitor().log("OUT_OF_ORDER owner create on {}", idxElem.key);
			restoreOwner.restore(idxElem.key, new String(idxElem.payload));
			log.create(idxElem.key.type, idxElem.key);
		}

		try {
			restore.restore(event.key, payload);
		} catch (Throwable e) { // NOSONAR
			if (!ignoreFailure) {
				log.monitor().log("[{}:{}] Failure processing type {}", event.part, event.offset, event.key.type);
				log.failure(restore.type(), event.key, payload, e);
				throw e;
			} else {
				log.skip(event.key.type, event.key, payload);
			}
		}
	}

	private void fixupKey(RecordKey key) {
		String container = key.uid;
		key.uid = state.uidAlias(container);
	}

}
