/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.sds.store.cyrusspool;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.luben.zstd.RecyclingBufferPool;
import com.github.luben.zstd.ZstdInputStream;
import com.github.luben.zstd.ZstdOutputStream;
import com.google.common.base.Stopwatch;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.Unpooled;
import net.bluemind.backend.cyrus.partitions.CyrusFileSystemPathHelper;
import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.cyrus.partitions.MailboxDescriptor;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.replica.api.IDbReplicatedMailboxes;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.IReplicatedMailboxesMgmt;
import net.bluemind.backend.mail.replica.api.MailboxRecordItemUri;
import net.bluemind.backend.mail.replica.api.Tier;
import net.bluemind.backend.mail.replica.api.TierMove;
import net.bluemind.config.DataLocation;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.hornetq.client.MQ.SharedMap;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.lib.jutf7.UTF7Converter;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.network.topology.Topology;
import net.bluemind.node.api.INodeClient;
import net.bluemind.node.api.LocalNodeClient;
import net.bluemind.node.api.NodeActivator;
import net.bluemind.sds.dto.DeleteRequest;
import net.bluemind.sds.dto.ExistRequest;
import net.bluemind.sds.dto.ExistResponse;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.PutRequest;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.dto.TierMoveRequest;
import net.bluemind.sds.dto.TierMoveResponse;
import net.bluemind.sds.store.ISdsBackingStore;
import net.bluemind.sds.store.PathHandler;
import net.bluemind.server.api.Server;
import net.bluemind.system.api.ArchiveKind;
import net.bluemind.system.api.SysConfKeys;

public class SpoolBackingStore implements ISdsBackingStore {

	private static final Logger logger = LoggerFactory.getLogger(SpoolBackingStore.class);
	private final IServiceProvider serviceProvider;
	private final SharedMap<String, String> sharedMap = Shared.mapSysconf();
	private final INodeClient nc;

	public SpoolBackingStore(IServiceProvider prov, ItemValue<Server> backend) {
		this.serviceProvider = prov;
		if (DataLocation.current().equals(backend.uid)) {
			this.nc = new LocalNodeClient();
		} else {
			this.nc = NodeActivator.get(backend.value.address());
		}

	}

	@Override
	public CompletableFuture<SdsResponse> upload(PutRequest req) {
		var chrono = Stopwatch.createStarted();
		String target = chooseTarget(req);
		long tgtSel = ms(chrono);
		long comp = 0L;
		ByteBuf bb = Unpooled.buffer(2 * (int) new File(req.filename).length());
		PathHandler ph = PathHandler.forPathOrUri(req.filename);
		try (InputStream input = ph.read();
				ByteBufOutputStream bbo = new ByteBufOutputStream(bb);
				OutputStream zst = new ZstdOutputStream(bbo, RecyclingBufferPool.INSTANCE, -3)) {
			long copied = input.transferTo(zst);
			logger.debug("Compressed {}byte(s) for {}", copied, req.guid);
			comp = ms(chrono);
		} catch (IOException e) {
			return CompletableFuture.failedFuture(e);
		}
		try (InputStream input = new ByteBufInputStream(bb, true)) {
			nc.writeFile(target, input);
			long upload = ms(chrono);
			if (upload > 500) {
				logger.info("{} stored. Timings (select: {}ms, comp: {}ms, upload: {}ms)", req.guid, tgtSel, comp,
						upload);
			}
			return CompletableFuture.completedFuture(SdsResponse.UNTAGGED_OK);
		} catch (IOException e) {
			return CompletableFuture.failedFuture(e);
		}

	}

	private long ms(Stopwatch s) {
		long ret = s.elapsed(TimeUnit.MILLISECONDS);
		s.reset().start();
		return ret;
	}

	/*
	 * Returns the path to store the new message, watching deliveryDate to choose
	 * the correct storage tier
	 */
	private String chooseTarget(PutRequest req) {
		if (req.deliveryDate == null) {
			return livePath(req.guid);
		}
		ArchiveKind archiveKind = ArchiveKind.fromName(sharedMap.get(SysConfKeys.archive_kind.name()));
		Integer archiveDays;
		try {
			archiveDays = Integer
					.parseInt(Optional.ofNullable(sharedMap.get(SysConfKeys.archive_days.name())).orElse("0"));
		} catch (NumberFormatException nfe) {
			archiveDays = 0;
		}
		if (archiveKind != null && archiveKind.supportsHsm() && archiveDays > 0
				&& req.deliveryDate.toInstant().isBefore(Instant.now().minus(archiveDays, ChronoUnit.DAYS))) {
			// deliveryDate is before now() - tierChangeDelay => direct insert to archive
			return archivePath(req.guid);
		} else {
			return livePath(req.guid);
		}
	}

	@Override
	public CompletableFuture<ExistResponse> exists(ExistRequest req) {
		return locateGuid(null, req.guid, false).thenApply(sdsResp -> ExistResponse.from(sdsResp.succeeded()));
	}

	@Override
	public CompletableFuture<SdsResponse> download(GetRequest req) {
		return locateGuid(PathHandler.forPathOrUri(req.filename), req.guid, true);
	}

	@Override
	public CompletableFuture<SdsResponse> downloadRaw(GetRequest req) {
		return locateGuid(PathHandler.forPathOrUri(req.filename), req.guid, false);
	}

	private CompletableFuture<SdsResponse> locateGuid(PathHandler targetPath, String guid, boolean decompress) {
		// check new live path
		String path = livePath(guid);
		try {
			LocatePathWithSize locatePath = locatePath(targetPath, path, decompress);
			if (locatePath.exists()) {
				return CompletableFuture.completedFuture(SdsResponse.sdsResponseWithSize(locatePath.size()));
			}
		} catch (IOException ie) {
			return CompletableFuture.completedFuture(SdsResponse.error(ie, guid, true));
		}

		// check new archive path
		path = archivePath(guid);
		try {
			LocatePathWithSize locatePath = locatePath(targetPath, path, decompress);
			if (locatePath.exists()) {
				return CompletableFuture.completedFuture(SdsResponse.sdsResponseWithSize(locatePath.size()));
			}
		} catch (IOException ie) {
			return CompletableFuture.completedFuture(SdsResponse.error(ie, guid, true));
		}

		/*
		 * If sds_cyrus_spool_enabled is null, it means the sds spool upgraded is not
		 * yet finished, so try getting emails from the old cyrus spool.
		 */
		if (Boolean.TRUE.equals(Boolean.valueOf(
				Optional.ofNullable(sharedMap.get(SysConfKeys.sds_cyrus_spool_enabled.name())).orElse("true")))) {
			// check old cyrus stuff
			IReplicatedMailboxesMgmt mgmtApi = serviceProvider.instance(IReplicatedMailboxesMgmt.class);
			Set<MailboxRecordItemUri> refs = mgmtApi.getBodyGuidReferences(guid);
			if (!refs.isEmpty()) {
				MailboxRecordItemUri uri = refs.iterator().next();
				IContainers contApi = serviceProvider.instance(IContainers.class);
				BaseContainerDescriptor cont = contApi.getLightIfPresent(uri.containerUid);
				IMailboxes mboxes = serviceProvider.instance(IMailboxes.class, cont.domainUid);
				ItemValue<Mailbox> mbox = mboxes.getComplete(uri.owner);
				if (mbox == null) {
					logger.error("Mailbox of {} not found. Deleted?", uri.owner);
				} else {
					IDbReplicatedMailboxes folderApi = serviceProvider.instance(IDbReplicatedMailboxes.class,
							cont.domainUid, mbox.value.type.nsPrefix + mbox.value.name);
					ItemValue<MailboxFolder> folder = folderApi
							.getComplete(IMailReplicaUids.uniqueId(uri.containerUid));
					if (folder != null) { // Broken user folder ?
						ItemValue<Server> server = Topology.get().datalocation(cont.datalocation);
						CyrusPartition part = CyrusPartition.forServerAndDomain(server, cont.domainUid);
						MailboxDescriptor desc = new MailboxDescriptor();
						desc.type = mbox.value.type;
						desc.mailboxName = mbox.value.name;
						desc.utf7FolderPath = UTF7Converter.encode(folder.value.fullName);
						path = CyrusFileSystemPathHelper.getFileSystemPath(cont.domainUid, desc, part, uri.imapUid);
						try {
							LocatePathWithSize pathWithSize = onNode(targetPath, path, decompress);
							if (pathWithSize.exists()) {
								logger.debug("{} -> '{}'", guid, path);
								return CompletableFuture
										.completedFuture(SdsResponse.sdsResponseWithSize(pathWithSize.size()));
							}
						} catch (IOException ie) {
							return CompletableFuture.completedFuture(SdsResponse.error(ie, guid, true));
						}
						path = CyrusFileSystemPathHelper.getHSMFileSystemPath(cont.domainUid, desc, part, uri.imapUid);
						try {
							LocatePathWithSize pathWithSize = onNode(targetPath, path, decompress);
							if (pathWithSize.exists()) {
								logger.debug("{} -> '{}'", guid, path);
								return CompletableFuture
										.completedFuture(SdsResponse.sdsResponseWithSize(pathWithSize.size()));
							}
						} catch (IOException ie) {
							return CompletableFuture.completedFuture(SdsResponse.error(ie, guid, true));
						}
					} else {
						logger.error("Folder {} not found. Deleted?", uri.containerUid);
					}
				}
			}
		}
		return CompletableFuture.completedFuture(SdsResponse.error(guid + " not found", guid, false));
	}

	@Override
	public CompletableFuture<TierMoveResponse> tierMove(TierMoveRequest tierMoveRequest) {
		logger.debug("Tier move request {}", tierMoveRequest);
		List<String> errors = new ArrayList<>();
		List<String> successes = new ArrayList<>();
		for (TierMove move : tierMoveRequest.moves) {
			String from;
			String to;

			if (move.tier.equals(Tier.SLOW)) {
				from = livePath(move.messageBodyGuid);
				to = archivePath(move.messageBodyGuid);
			} else {
				from = archivePath(move.messageBodyGuid);
				to = livePath(move.messageBodyGuid);
			}

			if (nc.exists(from)) {
				try {
					nc.moveFile(from, to);
					successes.add(move.messageBodyGuid);
				} catch (ServerFault e) {
					logger.error("TIERING failed to move {} to {}", from, to, e);
					errors.add(move.messageBodyGuid);
				}
			} else {
				// file is missing, retrying is pointless
				logger.warn("TIERING marking move of {} as successful but file is missing", from);
				successes.add(move.messageBodyGuid);
			}
		}
		return CompletableFuture.completedFuture(new TierMoveResponse(successes, errors));
	}

	private LocatePathWithSize locatePath(PathHandler targetPath, String emlPath, boolean decompress)
			throws IOException {
		return onNode(targetPath, emlPath, decompress);
	}

	private record LocatePathWithSize(PathHandler path, boolean exists, long size) {

	}

	private LocatePathWithSize onNode(PathHandler targetPath, String emlPath, boolean decompress) throws IOException {
		if (targetPath == null) {
			boolean exists = nc.exists(emlPath);
			return new LocatePathWithSize(targetPath, exists, 0L);
		} else {
			byte[] eml = nc.read(emlPath);
			if (eml.length > 0) {
				logger.debug("Found {} byte(s) of mail data in {}, tgt is {}{}", eml.length, emlPath, targetPath,
						(decompress ? "" : " (compressed)"));
				if (decompress && emlPath.endsWith(".zst")) {
					return compressedEml(targetPath, eml);
				} else {
					return plainEml(targetPath, eml);
				}
			}
		}
		return new LocatePathWithSize(targetPath, false, 0L);
	}

	private LocatePathWithSize plainEml(PathHandler targetPath, byte[] eml) throws IOException {
		try (OutputStream out = targetPath.openForWriting()) {
			out.write(eml);
		}
		logger.debug("Wrote plain {} byte(s) to {}", eml.length, targetPath);
		return new LocatePathWithSize(targetPath, true, eml.length);
	}

	private LocatePathWithSize compressedEml(PathHandler targetPath, byte[] eml) throws IOException {
		ByteBufInputStream oio = new ByteBufInputStream(Unpooled.wrappedBuffer(eml));
		try (ZstdInputStream in = new ZstdInputStream(oio, RecyclingBufferPool.INSTANCE);
				OutputStream out = targetPath.openForWriting()) {
			long copied = in.transferTo(out);
			out.flush();
			logger.debug("Wrote compressed {} byte(s) to {}", copied, targetPath);
			return new LocatePathWithSize(targetPath, true, copied);
		}
	}

	private String livePath(String guid) {
		return "/var/spool/cyrus/data/by_hash/" + guid.charAt(0) + "/" + guid.charAt(1) + "/" + guid + ".zst";
	}

	private String archivePath(String guid) {
		return "/var/spool/bm-hsm/data/by_hash/" + guid.charAt(0) + "/" + guid.charAt(1) + "/" + guid + ".zst";
	}

	@Override
	public CompletableFuture<SdsResponse> delete(DeleteRequest req) {
		for (String guid : req.guids) {
			try {
				nc.deleteFile(livePath(guid));
				nc.deleteFile(archivePath(guid));
			} catch (Exception e) {
				return CompletableFuture.failedFuture(e);
			}
		}
		return CompletableFuture.completedFuture(SdsResponse.UNTAGGED_OK);
	}

	@Override
	public void close() {
		// that's ok
	}

}
