/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.sdsspool;

import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import io.github.resilience4j.core.IntervalFunction;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import io.github.resilience4j.retry.RetryRegistry;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.network.topology.Topology;
import net.bluemind.sds.dto.GetRequest;
import net.bluemind.sds.dto.SdsError;
import net.bluemind.sds.dto.SdsResponse;
import net.bluemind.sds.store.ISdsBackingStore;
import net.bluemind.sds.store.ISdsBackingStoreFactory;
import net.bluemind.sds.store.loader.SdsStoreLoader;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.api.ArchiveKind;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.sysconf.helper.SysConfHelper;

public class SdsSpoolDownloader {
	private static final Logger logger = LoggerFactory.getLogger(SdsSpoolDownloader.class);
	// Map of serverUid / SdsStore
	private final Map<String, ISdsBackingStore> sdsStores = loadSdsStores();
	private final ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(0,
			Thread.ofVirtual().name("sds-spool-downloader-retry-").factory());
	private final Map<Transfer, CompletableFuture<SdsResponse>> inFlightTransfers;
	private final Semaphore inFlightSemaphore;
	private final Retry retry;

	/*
	 * Will receive an offset of the downloaded file with success and the minimum
	 * offset in queue
	 */
	public interface OffsetCommit {
		void commitOffsets(long transfer, long miniInQ, boolean nothingInFlight);
	}

	private final OffsetCommit commitOffsetConsumer;
	private final Consumer<String> stopExecutionConsumer;

	public static final int DEFAULT_CONCURRENCY = 32;

	private final LongAdder downloadedSize;

	public SdsSpoolDownloader(OffsetCommit commitOffsetConsumer, Consumer<String> stopExecutionConsumer) {
		this(commitOffsetConsumer, stopExecutionConsumer, DEFAULT_CONCURRENCY);
	}

	public SdsSpoolDownloader(OffsetCommit commitOffsetConsumer, Consumer<String> stopExecutionConsumer,
			int concurrency) {
		inFlightSemaphore = new Semaphore(concurrency);
		this.commitOffsetConsumer = commitOffsetConsumer;
		this.stopExecutionConsumer = stopExecutionConsumer;
		this.inFlightTransfers = new ConcurrentHashMap<>(concurrency * 2);
		this.downloadedSize = new LongAdder();

		Config coreConfig = CoreConfig.get();

		// 4 minutes of retry
		RetryConfig retryConfig = RetryConfig.<SdsResponse>custom() //
				.maxAttempts(coreConfig.getInt(CoreConfig.DataProtect.SDS_DOWNLOAD_RETRIES))
				.intervalFunction(IntervalFunction.ofExponentialBackoff(1000, 2)) //
				.retryExceptions(ServerFault.class, IOException.class) //
				.failAfterMaxAttempts(true) //
				.retryOnResult(sdsresp -> {
					if (sdsresp.error == null) {
						return false;
					} else {
						SdsError error = sdsresp.error;
						if (error.message.contains("No space left")) {
							return false;
						}
						return sdsresp.error.retryable;
					}
				}) //
				.build();
		retry = RetryRegistry.of(retryConfig).retry("sdsdownload", retryConfig);
	}

	public long getDownloadedSize() {
		return downloadedSize.longValue();
	}

	public void addTransfer(String serverUid, String guid, Path filepath, long offset) throws RefusedTransfer {
		ISdsBackingStore store = sdsStores.get(serverUid);
		if (store == null) {
			throw new RefusedTransfer("Unable to find a valid SDS store for serverUid '" + serverUid + "'");
		}
		processTransfer(new Transfer(store, guid, filepath, offset));
	}

	public long minimumOffsetInQueue() {
		return inFlightTransfers.entrySet().stream().map(Entry::getKey).mapToLong(Transfer::offset).min().orElse(0);
	}

	public int currentlyInTransfer() {
		return inFlightTransfers.size();
	}

	private void processTransfer(Transfer transfer) {
		inFlightSemaphore.acquireUninterruptibly();
		try {
			inFlightTransfers.put(transfer,
					retry.executeCompletionStage(scheduledExecutorService,
							() -> transfer.store()
									.downloadRaw(GetRequest.of(transfer.guid(), transfer.filepath().toString()))
									.whenComplete((r, t) -> {
										if (r.succeeded()) {
											downloadedSize.add(r.size());
											commitOffsetConsumer.commitOffsets(transfer.offset(),
													minimumOffsetInQueue(), inFlightTransfers.isEmpty());
										} else {
											logger.warn("Download of {} failed: {}", transfer.guid,
													r.error != null ? r.error.message : "");
										}
										/*
										 * We need this to avoid trying to remove a completablefuture which is not in
										 * the map yet
										 */
										scheduledExecutorService.schedule(() -> {
											var cf = inFlightTransfers.remove(transfer);
											if (cf == null) {
												logger.error(
														"Unable to remove inflight transfer {} from inFlightTransfers map with {}",
														transfer, inFlightTransfers.size());
											}
										}, 1, TimeUnit.SECONDS);
										inFlightSemaphore.release();
									}))
							.toCompletableFuture());
		} catch (CompletionException e) {
			logger.error("Exec error", e);
			inFlightSemaphore.release();
			stopExecutionConsumer.accept("unrecoverable download failure: " + e.getMessage());
		}
	}

	public CompletableFuture<Void> asPromise() {
		CompletableFuture<Void> fut = new CompletableFuture<>();
		Thread.ofVirtual().name("sds-downloader-end").start(() -> {
			while (!inFlightTransfers.isEmpty()) {
				try {
					Thread.sleep(200);
				} catch (InterruptedException e) { // NOSONAR
					fut.completeExceptionally(e);
				}
				inFlightTransfers.entrySet().removeIf(e -> e.getValue() != null && e.getValue().isDone());
			}
			fut.complete(null);
		});
		return fut;
	}

	private static Map<String, ISdsBackingStore> loadSdsStores() {
		Map<String, ISdsBackingStore> stores = new HashMap<>();
		SystemConf config = SysConfHelper.fromSharedMap();
		List<ISdsBackingStoreFactory> storeFactories = SdsStoreLoader.getUnpooledStoreFactories();
		for (ItemValue<Server> server : Topology.get().all(TagDescriptor.mail_imap.getTag())) {
			ArchiveKind storeType = archiveKind(config);
			if (storeType == null || !storeType.isSdsArchive()) {
				continue;
			}
			storeFactories.stream().filter(sbs -> sbs.kind() == storeType).findAny()
					.map(sf -> sf.create(VertxPlatform.getVertx(), config, server.uid))
					.ifPresent(store -> stores.put(server.uid, store));
		}
		return stores;
	}

	private static ArchiveKind archiveKind(SystemConf sysconf) {
		String archiveKind = Optional.ofNullable(sysconf.stringValue(SysConfKeys.archive_kind.name())).orElse("cyrus");
		if (archiveKind.isBlank() || archiveKind.equalsIgnoreCase("none")) {
			archiveKind = "cyrus";
		}
		return ArchiveKind.fromName(archiveKind);
	}

	public record Transfer(ISdsBackingStore store, String guid, Path filepath, long offset) {
		@Override
		public final String toString() {
			return "Transfer[store=" + store.getClass().getName() + "@" + Objects.hashCode(store) + ", g=" + guid
					+ ", o=" + offset + ", f=" + filepath + "]";
		}
	}

	@SuppressWarnings("serial")
	public static class RefusedTransfer extends Exception {
		RefusedTransfer(String message) {
			super(message);
		}
	}

}
