/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.directory.cql.store;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.BatchStatement;
import com.datastax.oss.driver.api.core.cql.BatchType;
import com.datastax.oss.driver.api.core.cql.ResultSet;
import com.datastax.oss.driver.api.core.data.UdtValue;
import com.datastax.oss.driver.api.core.type.UserDefinedType;
import com.datastax.oss.driver.api.core.type.codec.registry.MutableCodecRegistry;
import com.datastax.oss.driver.internal.core.type.codec.extras.enums.EnumNameCodec;
import com.datastax.oss.driver.shaded.guava.common.collect.Lists;

import net.bluemind.core.api.Email;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.cql.store.CqlItemStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.cql.CqlSessions;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.repository.DirEntryNG;
import net.bluemind.directory.repository.IDirEntryStore;
import net.bluemind.mailbox.api.Mailbox.Routing;
import net.bluemind.repository.sequences.SequenceStores;

public class CqlDirEntryNgStore extends CqlAbstractStore implements IDirEntryStore {

	private static final ConSetup conSetup = new ConSetup();
	private static final UserDefinedType email = loadEmailType();
	private static final UserDefinedType loginCaps = loadLoginCaps();
	private static final UserDefinedType signature = loadSignature();

	private static UserDefinedType loadEmailType() {
		return CqlSessions.forKeyspace("core").getMetadata().getKeyspace("core")
				.flatMap(ks -> ks.getUserDefinedType("dir_email"))
				.orElseThrow(() -> new IllegalArgumentException("dir_email def not found"));
	}

	private static UserDefinedType loadLoginCaps() {
		return CqlSessions.forKeyspace("core").getMetadata().getKeyspace("core")
				.flatMap(ks -> ks.getUserDefinedType("login_caps"))
				.orElseThrow(() -> new IllegalArgumentException("dir_email def not found"));
	}

	private static UserDefinedType loadSignature() {
		return CqlSessions.forKeyspace("core").getMetadata().getKeyspace("core")
				.flatMap(ks -> ks.getUserDefinedType("signature"))
				.orElseThrow(() -> new IllegalArgumentException("dir_email def not found"));
	}

	private final long contId;
	private final CqlItemStore itemStore;
	private String domainUid;

	public static class ConSetup {
		private AtomicBoolean once = new AtomicBoolean(false);

		public void runOnce() {
			if (once.compareAndSet(false, true)) {
				try {
					run();
				} catch (Throwable t) { // NOSONAR
					once.compareAndExchange(true, false);
				}
			}
		}

		private void run() {
			CqlSession pool = CqlSessions.forKeyspace("core");
			if (pool.getContext().getCodecRegistry() instanceof MutableCodecRegistry mcr) {
				mcr.register(new EnumNameCodec<>(Kind.class));
				mcr.register(new EnumNameCodec<>(Routing.class));
			} else {
				throw new UnsupportedOperationException("Cannot register custom codec");
			}
		}
	}

	public CqlDirEntryNgStore(CqlSession s, Container c) {
		super(s);
		conSetup.runOnce();
		this.contId = c.id;
		this.domainUid = c.uid;
		this.itemStore = new CqlItemStore(c, s, SequenceStores.getDefault(), SecurityContext.SYSTEM);
	}

	@Override
	public void create(Item item, DirEntryNG value) throws SQLException {
		store(item, value);
	}

	private void store(Item item, DirEntryNG value) {
		DirEntryNG previous = get(item);
		if (previous != null && previous.emails != null) {
			for (Email e : previous.emails) {
				voidCql("DELETE FROM t_dir_email WHERE container_id=? AND local_part=? AND domain_part=?", contId,
						e.localPart(), domPart(e));
			}
		}
		UdtValue authCaps = null;
		if (value.loginCaps != null) {
			var lc = value.loginCaps;
			authCaps = loginCaps.newValue(lc.passwordHash, lc.lastChange, lc.mustChange, lc.neverExpires);
		}

		List<UdtValue> mails = Optional.ofNullable(value.emails).orElse(Collections.emptyList()).stream()
				.map(e -> email.newValue(e.localPart(), e.allAliases ? null : e.domainPart(), e.isDefault)).toList();
		BatchStatement batch = BatchStatement.newInstance(BatchType.LOGGED);
		batch = batch.add(b("""
				INSERT INTO t_dir_entry
				(kind, display_name, emails, auth_capabilities, mailbox_name, routing, container_id, item_id)
				VALUES
				(?, ?, ?, ?, ?, ?, ?, ?)
				""", value.kind, item.displayName, mails, authCaps, value.mailboxName, value.routing, contId, item.id));
		if (value.emails != null) {
			for (Email e : value.emails) {
				batch = batch.add(
						b("INSERT INTO t_dir_email (container_id, local_part, domain_part, item_id) values (?,?,?,?)",
								contId, e.localPart(), domPart(e), item.id));
			}
		}
		ResultSet res = session.execute(batch);
		if (!res.wasApplied()) {
			throw new ServerFault("nothing was applied for " + item);
		}
	}

	private String domPart(Email e) {
		return e.allAliases ? "ANY" : e.domainPart();
	}

	@Override
	public void update(Item item, DirEntryNG value) throws SQLException {
		store(item, value);
	}

	@Override
	public void delete(Item item) throws SQLException {
		voidCql("delete from t_dir_entry where container_id=? and item_id=?", contId, item.id);
	}

	private static final EntityPopulator<DirEntryNG> POP = (r, i, v) -> {
		v.itemId = r.getLong(i++);
		v.kind = r.get(i++, Kind.class);
		v.displayName = r.getString(i++);
		v.mailboxName = r.getString(i++);
		List<UdtValue> mailUdts = r.getList(i++, UdtValue.class);
		v.emails = mailUdts.stream().map(u -> {
			String local = u.getString(0);
			String dom = u.getString(1);
			boolean defaultEmail = u.getBoolean(2);
			boolean allAlias = dom == null;
			return Email.create(local + "@" + dom, defaultEmail, allAlias);
		}).toList();
		v.email = v.emails.stream().filter(e -> e.isDefault).findAny().map(e -> e.address).orElse(null);

		UdtValue authCaps = r.getUdtValue(i++);
		if (authCaps != null) {
			v.loginCaps = new DirEntryNG.LoginCaps();
			v.loginCaps.passwordHash = authCaps.getString(0);
			v.loginCaps.lastChange = authCaps.getInstant(1);
			v.loginCaps.mustChange = authCaps.getBoolean(2);
			v.loginCaps.neverExpires = authCaps.getBoolean(3);
		}

		v.routing = r.get(i++, Routing.class);
		v.memberOf = r.getSet(i++, String.class);
		return i;
	};

	@Override
	public DirEntryNG get(Item item) {
		DirEntryNG baseEntry = unique("""
				select item_id, kind, display_name, mailbox_name, emails, auth_capabilities, routing,
				member_of
				from t_dir_entry
				where container_id=? and item_id=?
				""", r -> new DirEntryNG(), POP, contId, item.id);
		if (baseEntry != null) {
			baseEntry.entryUid = item.uid;
			baseEntry.path = domainUid + "/" + baseEntry.kind.pathComponent() + "/" + baseEntry.entryUid;
		}
		return baseEntry;
	}

	@Override
	public void deleteAll() throws SQLException {
		// no need
	}

	@Override
	public List<DirEntryNG> getMultiple(List<Item> items) throws SQLException {
		Map<Long, String> idToUid = items.stream().collect(Collectors.toMap(iv -> iv.id, iv -> iv.uid));
		List<Long> asIds = items.stream().map(it -> it.id).toList();
		List<DirEntryNG> ret = new ArrayList<>(asIds.size());
		for (var slice : Lists.partition(asIds, 64)) {
			var chunk = map("""
					select item_id, kind, display_name, mailbox_name, emails, auth_capabilities, routing,
					member_of
					from t_dir_entry
					where container_id=? and item_id IN ?
					""", r -> new DirEntryNG(), POP, contId, slice).stream().map(de -> {
				de.entryUid = idToUid.get(de.itemId);
				de.path = domainUid + "/" + de.kind.pathComponent() + "/" + de.entryUid;
				return de;
			}).toList();
			ret.addAll(chunk);
		}
		return ret;
	}

	public Item byMailboxName(String name) {
		Long itemId = unique("SELECT item_id from t_dir_mailboxes where container_id=? and mailbox_name=?",
				r -> r.getLong(0), voidPop(), contId, name);
		if (itemId != null) {
			return itemStore.getById(itemId);
		} else {
			return null;
		}
	}

	public Item byEmail(String email) {
		String[] parts = email.split("@");
		Long itemId = unique(
				"SELECT item_id from t_dir_email where container_id=? and local_part=? and domain_part in ?",
				r -> r.getLong(0), voidPop(), contId, parts[0], List.of(parts[1], "ANY"));
		System.err.println("found " + itemId);
		if (itemId != null) {
			return itemStore.getById(itemId);
		} else {
			return null;
		}
	}

	public Item byOwnerUid(String owner) {
		return itemStore.get(owner);
	}

	public List<ItemIdAndRole> byKind(Kind k) {
		return map("select item_id, roles from t_dir_kind where container_id=? and kind=?",
				r -> new ItemIdAndRole(r.getLong(0), r.getSet(1, String.class)), voidPop(), contId, k);
	}

	public List<String> mailboxUids() {
		List<Long> mboxes = map("select item_id from t_dir_routings where container_id=? and routing in ?",
				r -> r.getLong(0), voidPop(), contId, List.of(Routing.internal, Routing.external));
		return itemStore.getMultipleById(mboxes).stream().map(i -> i.uid).toList();
	}

	@Override
	public boolean exists(Item item) throws SQLException {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public void addMemberOf(Item deItem, String groupUid) {
		batch(//
				b("update t_dir_entry set member_of = member_of + ? where container_id=? and item_id=?", //
						Set.of(groupUid), contId, deItem.id), //
				b("insert into t_direct_members (container_id, group_uid, item_id) values (?,?,?)", contId, groupUid,
						deItem.id)//
		);
	}

	@Override
	public void rmMemberOf(Item deItem, String groupUid) {
		batch(//
				b("update t_dir_entry set member_of = member_of - ? where container_id=? and item_id=?", //
						Set.of(groupUid), contId, deItem.id), //
				b("delete from t_direct_members where container_id=? and group_uid=? and item_id=?", contId, groupUid,
						deItem.id)//
		);
	}

}
