/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.milter.action.misc.updatesubject;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;

public class EmailHeaderEncoder {

	private EmailHeaderEncoder() {

	}

	public static String encodeHeaderValue(String input) {
		if (input == null || input.isEmpty()) {
			return input;
		}

		if (containsNonAscii(input)) {
			return replaceNonAsciiChars(input);
		}

		return input;
	}

	public static boolean containsNonAscii(String input) {
		if (input.contains("=?") && input.contains("?=")) {
			return false; // already encoded
		}
		for (char c : input.toCharArray()) {
			if (c > 127) {
				return true;
			}
		}
		return false;
	}

	private static String replaceNonAsciiChars(String input) {
		if (input == null)
			return null;

		StringBuilder result = new StringBuilder();

		List<Integer> buffer = new ArrayList<>();
		for (char c : input.toCharArray()) {

			if (c <= 127) {
				handleBuffer(result, buffer);
				switch (c) {
				case 32 -> result.append('_'); // SPACE is encoded as '_'
				case 95 -> result.append("=5F"); // unserscore is encoded
				default -> result.append(c);
				}
				buffer.clear();
			} else {
				int unsignedInt = Byte.toUnsignedInt((byte) c);
				if (unsignedInt == 233) { // ugly workaround to fix "éé", like in "créé"
					buffer.add(unsignedInt);
					handleBuffer(result, buffer);
					buffer.clear();
				} else {
					buffer.add(unsignedInt);
				}
			}
		}
		handleBuffer(result, buffer);
		if (!result.toString().equals(input)) {
			return "=?UTF-8?Q?" + result.toString() + "?=";
		}

		return input;
	}

	private static void handleBuffer(StringBuilder result, List<Integer> buffer) {
		if (!buffer.isEmpty()) {
			char[] chars = new char[buffer.size()];
			for (int i = 0; i < chars.length; i++) {
				chars[i] = ((char) buffer.get(i).intValue());
			}
			append(result, buffer, chars);
		}
	}

	private static void append(StringBuilder result, List<Integer> buffer, char[] chars) {
		byte[] utf8Bytes = String.valueOf(chars)
				.getBytes(buffer.size() > 1 ? StandardCharsets.ISO_8859_1 : StandardCharsets.UTF_8);
		for (byte b : utf8Bytes) {
			result.append(String.format("=%02X", b & 0xFF));
		}
	}
}