/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.imap.vt.cmd;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.google.common.base.Splitter;

import io.vertx.core.buffer.Buffer;
import net.bluemind.imap.vt.parsing.IncomingChunk;

public class UidSearchCommand extends TaggedCommand<List<Long>> {

	private final String idSet;

	public UidSearchCommand(CommandContext ctx, String idSetAndQuery) {
		super(ctx);
		this.idSet = idSetAndQuery;
	}

	@Override
	protected void buildCommand(Buffer b) {
		b.appendString("UID SEARCH " + idSet);
	}

	private static final Pattern fetched = Pattern.compile("\\* SEARCH (.*)$");
	private static final Splitter FL_SPLIT = Splitter.on(' ').omitEmptyStrings();

	@Override
	protected List<Long> processChunks(List<IncomingChunk> chunks) throws IOException {
		var last = chunks.getLast();
		for (var ic : chunks) {
			if (ic == last) {
				break;
			}
			String t = ic.pieces().getFirst().txt();
			Matcher m = fetched.matcher(t);
			if (m.find()) {
				return FL_SPLIT.splitToStream(m.group(1)).map(Long::parseLong).toList();
			}
		}
		return Collections.emptyList();
	}

}
