/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal;

import java.io.IOException;
import java.lang.ref.Reference;
import java.sql.SQLException;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.CountingInputStream;

import io.netty.buffer.ByteBufUtil;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.eventbus.DeliveryOptions;
import io.vertx.core.eventbus.MessageProducer;
import io.vertx.core.streams.ReadStream;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.MessageBody.Header;
import net.bluemind.backend.mail.parsing.BodyStreamProcessor;
import net.bluemind.backend.mail.parsing.BodyStreamProcessor.MessageBodyData;
import net.bluemind.backend.mail.replica.api.IMessageBodyTierChange;
import net.bluemind.backend.mail.replica.service.IInternalDbMessageBodies;
import net.bluemind.backend.mail.replica.service.sds.MessageBodyObjectStore;
import net.bluemind.backend.mail.repository.IMessageBodyStore;
import net.bluemind.common.telemetry.EmailTracer;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.vertx.VertxStream;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.lib.vertx.utils.StreamToChunk;
import net.bluemind.memory.pool.api.ChunkFiler;
import net.bluemind.sds.sync.api.SdsSyncEvent;
import net.bluemind.sds.sync.api.SdsSyncEvent.Body;
import net.bluemind.size.helper.MaxMessageSize;

public class DbMessageBodiesService implements IInternalDbMessageBodies {

	private static final Logger logger = LoggerFactory.getLogger(DbMessageBodiesService.class);

	protected final IMessageBodyStore bodyStore;

	private final Supplier<MessageBodyObjectStore> bodyObjectStore;
	private final Supplier<IMessageBodyTierChange> bodyTierChangeService;
	private final MessageProducer<Body> bodyAddPublisher;

	public DbMessageBodiesService(IMessageBodyStore bodyStore, Supplier<MessageBodyObjectStore> bodyObjectStore,
			Supplier<IMessageBodyTierChange> bodyTierChangeService) {
		this.bodyStore = bodyStore;
		this.bodyObjectStore = bodyObjectStore;
		this.bodyTierChangeService = bodyTierChangeService;
		bodyAddPublisher = VertxPlatform.eventBus().publisher(SdsSyncEvent.BODYADD.busName());
		bodyAddPublisher.deliveryOptions(new DeliveryOptions().setLocalOnly(true).setCodecName("SdsSyncBodyCodec"));
	}

	@Override
	public void create(String uid, Stream pristine) {
		create0(uid, new Date(), pristine);
	}

	@Override
	public void createWithDeliveryDate(String uid, long deliveryDate, Stream pristine) {
		create0(uid, new Date(deliveryDate), pristine);
	}

	/*
	 * deliveryDate is used to choose the correct storage tier
	 */
	private void create0(String uid, Date deliveryDate, Stream pristine) {
		if (exists(uid)) {
			logger.debug("Skipping existing body {}", uid);
			VertxStream.sink(pristine).orTimeout(10, TimeUnit.SECONDS).join();
			return;
		}

		ReadStream<Buffer> classic = VertxStream.read(pristine);
		String chunkAddr = StreamToChunk.filerAddress(classic, MaxMessageSize.get());

		MessageBodyObjectStore objectStore = bodyObjectStore.get();
		try (CountingInputStream inEml = new CountingInputStream(ChunkFiler.byAddress(chunkAddr).openStream())) {
			objectStore.store(uid, deliveryDate, chunkAddr);
			if (parseAndIndex(uid, deliveryDate, inEml) > 0) {
				bodyAddPublisher.write(new Body(ByteBufUtil.decodeHexDump(uid), objectStore.dataLocation()));
			}
		} catch (IOException e) {
			BodiesCache.bodies.invalidate(uid);
			logger.error("Error trying to store message body {}", uid, e);
			throw new ServerFault(e);
		} catch (Throwable t) { // NOSONAR
			BodiesCache.bodies.invalidate(uid);
			logger.error("Error trying to store message body {}", uid, t);
			throw t;
		} finally {
			ChunkFiler.byAddress(chunkAddr).release();
			Reference.reachabilityFence(chunkAddr);
		}
	}

	private int parseAndIndex(String uid, Date deliveryDate, CountingInputStream eml) {
		MessageBodyData bodyData = BodyStreamProcessor.parseBodyGetFullContent(eml, false);
		MessageBody body = bodyData != null ? bodyData.body : null;
		if (body != null) {
			logger.debug("Got body '{}'", body.subject);
			body.guid = uid;
			body.created = deliveryDate == null ? new Date() : deliveryDate;
			return updateAndIndex(bodyData);
		}
		return 0;
	}

	@Override
	public int updateAndIndex(MessageBodyData bodyData) {
		return update(bodyData.body);
	}

	@Override
	public void delete(String uid) {
		throw new ServerFault("NO: bodies are removed by reference counting");
	}

	public MessageBody getComplete(String uid) {
		return BodiesCache.bodies.get(uid, t -> {
			try {
				return bodyStore.get(t);
			} catch (SQLException e) {
				throw new ServerFault(e);
			}
		});
	}

	@Override
	public boolean exists(String uid) {
		MessageBody existing = BodiesCache.bodies.getIfPresent(uid);
		if (existing != null) {
			return true;
		}
		try {
			return bodyStore.exists(uid);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public Integer update(MessageBody mb) {
		int affectedRows = 0;
		try {
			affectedRows = bodyStore.store(mb);
			if (affectedRows > 0) {
				BodiesCache.bodies.invalidate(mb.guid);
				bodyTierChangeService.get().createBody(mb);
			}
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
		return affectedRows;
	}

	@Override
	public List<MessageBody> multiple(List<String> uid) {
		try {
			return bodyStore.multiple(uid);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}

	}

	@Override
	public MessageBody get(String uid) {
		return getComplete(uid);
	}

	@Override
	public void restore(ItemValue<MessageBody> item, boolean isCreate) {
		if (item.value != null) {
			if (item.value.headers != null) {
				item.value.headers.stream().filter(h -> h.name.equals(EmailTracer.TRACE_HEADER_NAME))
						.map(Header::firstValue).findAny().ifPresent(traceparent -> EmailTracer.trace("body-restore",
								traceparent, item.value.messageId, item.value.subject));
			}
			// Update but without messing with tier change queue. On conflict, do NOT update
			try {
				int affectedRows = bodyStore.store(item.value);
				if (affectedRows > 0) {
					MessageBodyObjectStore objectStore = bodyObjectStore.get();
					bodyAddPublisher
							.write(new Body(ByteBufUtil.decodeHexDump(item.value.guid), objectStore.dataLocation()));
					BodiesCache.bodies.invalidate(item.value.guid);
				}
			} catch (SQLException e) {
				throw ServerFault.sqlFault(e);
			}
		}
	}

	@Override
	public ItemValueExists itemValueExists(String uid) {
		try {
			return new ItemValueExists(true, bodyStore.exists(uid));
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public void deleteOrphans() {
		try {
			bodyStore.deleteNoRecordNoPurgeOrphanBodies();
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
