/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.service.internal;

import java.sql.SQLException;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.CharMatcher;

import net.bluemind.backend.mail.api.FolderCounters;
import net.bluemind.backend.mail.api.IBaseMailboxFolders;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MailboxFolderSearchQuery;
import net.bluemind.backend.mail.api.SearchResult;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.backend.mail.replica.api.MailboxReplicaRootDescriptor;
import net.bluemind.backend.mail.replica.api.MailboxReplicaRootDescriptor.Namespace;
import net.bluemind.backend.mail.replica.service.names.INameSanitizer;
import net.bluemind.backend.mail.replica.service.names.NameSanitizers;
import net.bluemind.backend.mail.repository.IMailboxReplicaStore;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemChangelog;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.repository.IAclStore;
import net.bluemind.core.container.repository.IContainerStore;
import net.bluemind.core.container.service.ChangeLogUtil;
import net.bluemind.core.container.service.internal.ContainerStoreService;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.rest.BmContext;
import net.bluemind.index.MailIndexActivator;
import net.bluemind.lib.jutf7.UTF7Converter;
import net.bluemind.mailbox.api.IMailboxAclUids;

public class BaseReplicatedMailboxesService implements IBaseMailboxFolders {

	private static final Logger logger = LoggerFactory.getLogger(BaseReplicatedMailboxesService.class);

	protected final BmContext context;
	protected final IMailboxReplicaStore replicaStore;
	protected final ContainerStoreService<MailboxReplica> storeService;
	protected final IContainerStore contStore;

	protected final IAclStore aclStore;
	protected final Container container;

	protected final MailboxReplicaRootDescriptor root;
	protected final String dataLocation;
	protected final RBACManager rbac;

	protected final INameSanitizer nameSanitizer;

	public BaseReplicatedMailboxesService(MailboxReplicaRootDescriptor root, Container cont, BmContext context,
			IMailboxReplicaStore store, ContainerStoreService<MailboxReplica> mboxReplicaStore,
			IContainerStore contStore, IAclStore aclStore) {
		this.root = root;
		this.container = cont;
		this.context = context;
		this.replicaStore = store;
		this.storeService = mboxReplicaStore;
		this.contStore = contStore;
		this.aclStore = aclStore;
		this.dataLocation = root.dataLocation;
		this.nameSanitizer = NameSanitizers.create(root, store, mboxReplicaStore);
		this.rbac = RBACManager.forContext(context).forContainer(IMailboxAclUids.uidForMailbox(container.owner));
	}

	public ItemValue<MailboxFolder> root() {
		ItemValue<MailboxFolder> res;
		if (root.ns == Namespace.shared) {
			res = byName(root.name.replace('^', '.'));
		} else {
			res = byName("INBOX");
		}
		if (res == null) {
			logger.error("Failed to find root of {}", root);
		}
		return res;
	}

	public ItemValue<MailboxFolder> trash() {
		ItemValue<MailboxFolder> res;
		if (root.ns == Namespace.shared) {
			res = byName(root.name.replace('^', '.') + "/Trash");
		} else {
			res = byName("Trash");
		}
		if (res == null) {
			logger.error("Failed to find root of {}", root);
		}
		return res;
	}

	protected ItemValue<MailboxFolder> adapt(ItemValue<MailboxReplica> rec) {
		if (rec == null) {
			return null;
		}
		return ItemValue.create(rec, rec.value);
	}

	protected String decodeIfUTF7(String s) {
		if (CharMatcher.ascii().matchesAllOf(s)) {
			try {
				return UTF7Converter.decode(s);
			} catch (Error err) { // NOSONAR
				// because jutf7 does not honor onMalformedInput(REPLACE) and
				// Charset.decode
				// throws an Error in that case
				if (logger.isDebugEnabled()) {
					logger.debug("{} looks like utf-7 but it is not", s);
				}
				return s;
			}
		} else {
			if (logger.isDebugEnabled()) {
				logger.debug("{} contains non-ascii chars, not decoding as utf7.", s);
			}
			return s;
		}
	}

	protected ItemValue<MailboxReplica> byReplicaName(String name) {
		String uid = null;
		try {
			uid = replicaStore.byName(decodeIfUTF7(name));
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
		if (uid == null) {
			return null;
		} else {
			return getCompleteReplica(uid);
		}
	}

	@Override
	public ItemValue<MailboxFolder> byName(String name) {
		rbac.check(Verb.Visible.name());

		ItemValue<MailboxReplica> fetched = byReplicaName(name);
		if (fetched == null) {
			return null;
		} else {
			return adapt(fetched);
		}
	}

	@Override
	public ItemValue<MailboxFolder> getComplete(String uid) {
		rbac.check(Verb.Visible.name());

		ItemValue<MailboxReplica> fetched = getCompleteReplica(uid);
		return adapt(fetched);
	}

	public ItemValue<MailboxFolder> getLight(String uid) {
		return adapt(storeService.getLight(uid, null));
	}

	protected ItemValue<MailboxReplica> getCompleteReplica(String uid) {
		return Optional.ofNullable(MboxReplicasCache.byUid(uid)).orElseGet(() -> {
			ItemValue<MailboxReplica> fetched = storeService.get(uid, null);
			if (fetched != null && fetched.value != null) {
				fetched.value.dataLocation = dataLocation;
				MboxReplicasCache.cache(fetched);
			}
			return fetched;
		});
	}

	@Override
	public List<ItemValue<MailboxFolder>> all() {
		rbac.check(Verb.Visible.name());
		return storeService.all().stream().map(this::adapt).collect(Collectors.toList());
	}

	@Override
	public List<ItemValue<MailboxFolder>> childrensOf(String folderUid) {
		rbac.check(Verb.Visible.name());
		try {
			return storeService.getMultipleById(replicaStore.childrensOf(folderUid)).stream().map(this::adapt).toList();
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public ItemChangelog itemChangelog(String itemUid, Long since) throws ServerFault {
		rbac.check(Verb.Visible.name());

		return ChangeLogUtil.getItemChangeLog(itemUid, since, context, container);
	}

	@Override
	public ContainerChangeset<String> changeset(Long since) throws ServerFault {
		rbac.check(Verb.Visible.name());

		return storeService.changeset(since, Long.MAX_VALUE);
	}

	@Override
	public ContainerChangeset<Long> changesetById(Long since) throws ServerFault {
		rbac.check(Verb.Visible.name());

		return storeService.changesetById(since, Long.MAX_VALUE);
	}

	@Override
	public ContainerChangeset<ItemVersion> filteredChangesetById(Long since, ItemFlagFilter filter) throws ServerFault {
		rbac.check(Verb.Visible.name());
		return storeService.changesetById(since, filter);
	}

	@Override
	public long getVersion() throws ServerFault {
		rbac.check(Verb.Visible.name());
		return storeService.getVersion();
	}

	@Override
	public SearchResult searchItems(MailboxFolderSearchQuery query) {
		rbac.check(Verb.Visible.name());

		return MailIndexActivator.getService().searchItems(container.domainUid, container.owner,
				SearchQueryAdapter.adapt(context, container.domainUid, container.owner, query));
	}

	protected List<FolderCounters> folderCounters(List<Long> folderItemIds) {
		rbac.check(Verb.Visible.name());

		if (folderItemIds == null) {
			throw new ServerFault("folder list required");
		}
		int len = folderItemIds.size();
		if (len > 500) {
			throw new ServerFault("Can't load more than 500 counters (asked for " + len + ")");
		}

		try {
			return replicaStore.counters(folderItemIds);
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

	@Override
	public String getFullName(String uid) {
		ItemValue<MailboxReplica> cached = MboxReplicasCache.byUid(uid);
		if (cached != null && cached.value != null) {
			return cached.value.fullName;
		}
		try {
			MailboxReplica replica = replicaStore.get(Item.create(uid, 0));
			return replica != null ? replica.fullName : null;
		} catch (SQLException e) {
			throw ServerFault.sqlFault(e);
		}
	}

}
