/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2020
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.eas.dto.sync;

import java.util.Iterator;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Splitter;

public class CollectionId {

	public static final String SEPARATOR = "__";
	public static final Splitter splitter = Splitter.on(SEPARATOR);

	private final Optional<Long> subscriptionId;
	private final long folderId;

	private CollectionId(Optional<Long> subscriptionId, long folderId) {
		this.subscriptionId = subscriptionId;
		this.folderId = folderId;
	}

	public static CollectionId of(long subscriptionId, String folderId) {
		return new CollectionId(subscriptionId > 0 ? Optional.of(subscriptionId) : Optional.empty(),
				Long.parseLong(folderId));
	}

	public static CollectionId of(String collectionId) {
		Optional<Long> subscriptionId = Optional.empty();
		String folderId = collectionId;
		if (folderId.contains(SEPARATOR)) {
			Iterator<String> colIdIterator = splitter.split(folderId).iterator();
			subscriptionId = Optional.of(Long.parseLong(colIdIterator.next()));
			folderId = colIdIterator.next();
		}

		return new CollectionId(subscriptionId, Long.parseLong(folderId));
	}

	public Optional<Long> getSubscriptionId() {
		return subscriptionId;
	}

	public long getFolderId() {
		return folderId;
	}

	public String getValue() {
		if (!subscriptionId.isPresent()) {
			return Long.toString(folderId);
		}

		return subscriptionId.get() + SEPARATOR + folderId;
	}

	@Override
	public int hashCode() {
		return Objects.hash(folderId, subscriptionId);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		CollectionId other = (CollectionId) obj;
		return folderId == other.folderId && Objects.equals(subscriptionId, other.subscriptionId);
	}

	@Override
	public String toString() {
		return "CollectionId [subscriptionId=" + subscriptionId + ", folderId=" + folderId + "]";
	}

}
