/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.launcher;

import java.io.PrintStream;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import picocli.AutoComplete.GenerateCompletion;
import picocli.CommandLine;
import picocli.CommandLine.Command;
import picocli.CommandLine.Help;
import picocli.CommandLine.Help.Ansi;
import picocli.CommandLine.Help.ColorScheme;
import picocli.CommandLine.Model.CommandSpec;
import picocli.CommandLine.Model.OptionSpec;
import picocli.CommandLine.Model.PositionalParamSpec;
import picocli.CommandLine.Option;
import picocli.CommandLine.Spec;

@Command(mixinStandardHelpOptions = true, //
		subcommands = { HelpCommand.class, GenerateCompletion.class }, //
		exitCodeOnInvalidInput = 51, //
		exitCodeOnExecutionException = 50, //
		sortOptions = true) //
class ParentCommand implements Runnable {
	private static final Set<String> FILTERED_COMMANDS = Set.of("help", "generate-completion");

	@Spec
	CommandSpec spec;

	@Option(names = { "-h", "--help" }, description = "Affiche l'arbre complet des commandes.")
	boolean helpRequested;

	@Option(names = { "-v", "--verbose" }, description = "Affiche aussi les arguments pour chaque commande de l'arbre.")
	boolean verbose;

	private static final PrintStream out = System.out; // NOSONAR: ok

	@Override
	public void run() {
		if (helpRequested) {
			int maxWidth = calculateMaxCommandWidth(spec, 0);
			printCommandHierarchy(spec, 0, verbose, maxWidth);
		} else {
			spec.commandLine().usage(System.err); // NOSONAR: ok
		}
	}

	protected int calculateMaxCommandWidth(CommandSpec commandSpec, int depth) {
		int maxWidth = 0;

		if (depth > 0) {
			String indent = "  ".repeat(depth - 1);
			String commandNameToDisplay = commandSpec.name();
			CommandSpec parent = commandSpec.parent();
			if (parent != null && depth > 1) {
				commandNameToDisplay = parent.name() + " " + commandNameToDisplay;
			}
			String fullCommandText = indent + commandNameToDisplay;
			maxWidth = fullCommandText.length();
		}

		for (CommandLine subCommand : commandSpec.subcommands().values()) {
			CommandSpec subSpec = subCommand.getCommandSpec();
			if (shouldIncludeCommand(subSpec)) {
				int subWidth = calculateMaxCommandWidth(subSpec, depth + 1);
				maxWidth = Math.max(maxWidth, subWidth);
			}
		}

		return maxWidth;
	}

	protected void printCommandHierarchy(CommandSpec commandSpec, int depth, boolean verbose, int maxWidth) {
		if (depth > 0) {
			String indent = "  ".repeat(depth - 1);
			ColorScheme colorScheme = commandSpec.commandLine().getHelp().colorScheme();

			String commandNameToDisplay = commandSpec.name();
			CommandSpec parent = commandSpec.parent();
			if (parent != null && depth > 1) {
				commandNameToDisplay = parent.name() + " " + commandNameToDisplay;
			}

			String fullCommandText = indent + commandNameToDisplay;
			Ansi.Text styledName = colorScheme.commandText(fullCommandText);

			out.print(styledName);
			int paddingSize = Math.max(1, maxWidth - fullCommandText.length() + 2);
			for (var description : extractDescriptions(commandSpec)) {
				out.print(" ".repeat(paddingSize));
				out.println(description);
			}

			if (verbose) {
				String argumentsDetails = formatArgumentDetails(commandSpec, indent + "  ");
				if (!argumentsDetails.isEmpty()) {
					out.println(argumentsDetails);
				}
			}
		}
		for (CommandLine subCommand : commandSpec.subcommands().values()) {
			CommandSpec subSpec = subCommand.getCommandSpec();
			if (shouldIncludeCommand(subSpec)) {
				printCommandHierarchy(subSpec, depth + 1, verbose, maxWidth);
			}
		}
	}

	protected String formatArgumentDetails(CommandSpec commandSpec, String indent) {
		List<OptionSpec> options = commandSpec.options().stream().filter(o -> !o.hidden()).collect(Collectors.toList());
		List<PositionalParamSpec> positionals = commandSpec.positionalParameters().stream().filter(p -> !p.hidden())
				.collect(Collectors.toList());
		if (options.isEmpty() && positionals.isEmpty()) {
			return "";
		}

		int maxNameWidth = 0;
		for (OptionSpec opt : options) {
			String names = String.join(", ", opt.names());
			maxNameWidth = Math.max(maxNameWidth, (indent + names).length());
		}
		for (PositionalParamSpec pos : positionals) {
			String name = pos.paramLabel();
			maxNameWidth = Math.max(maxNameWidth, (indent + name).length());
		}

		ColorScheme colorScheme = commandSpec.commandLine().getHelp().colorScheme();
		Help.TextTable table = Help.TextTable.forColumns(colorScheme, //
				new Help.Column(maxNameWidth + 2, 0, Help.Column.Overflow.SPAN), //
				new Help.Column(80 - maxNameWidth, 0, Help.Column.Overflow.WRAP) //
		);

		for (OptionSpec opt : options) {
			String names = String.join(", ", opt.names());
			String desc = (opt.description() != null && opt.description().length > 0) ? opt.description()[0] : "";
			table.addRowValues(colorScheme.optionText(indent + names).toString(), desc);
		}
		for (PositionalParamSpec pos : positionals) {
			String name = pos.paramLabel();
			String desc = (pos.description() != null && pos.description().length > 0) ? pos.description()[0] : "";
			table.addRowValues(colorScheme.parameterText(indent + name).toString(), desc);
		}
		return table.toString();
	}

	protected boolean shouldIncludeCommand(CommandSpec spec) {
		return !FILTERED_COMMANDS.contains(spec.name()) && !spec.usageMessage().hidden();
	}

	protected List<String> extractDescriptions(CommandSpec spec) {
		return Arrays.stream(spec.usageMessage().description()).map(d -> d.replace("\n", " ")).toList();
	}

}