/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.group.hook;

import java.util.List;

import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.mailbox.api.internal.IInCoreMailboxes;
import net.bluemind.mailbox.api.rules.DelegationRule;

public class DefaultGroupHook implements IGroupHook {

	@Override
	public void onGroupCreated(GroupMessage created) throws ServerFault {
	}

	@Override
	public void onGroupUpdated(GroupMessage previous, GroupMessage current) throws ServerFault {
	}

	@Override
	public void onGroupDeleted(GroupMessage deleted) throws ServerFault {
		removeGroupFromDelegateRules(deleted.container.domainUid, deleted.group.uid);
	}

	@Override
	public void onAddMembers(GroupMessage group) throws ServerFault {
	}

	@Override
	public void onRemoveMembers(GroupMessage group) throws ServerFault {
	}

	private void removeGroupFromDelegateRules(String domainUid, String groupUid) {
		IInCoreMailboxes mailboxService = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IInCoreMailboxes.class, domainUid);

		getMailboxes(domainUid, groupUid) //
				.forEach(uid -> {
					DelegationRule mailboxDelegationRule = mailboxService.getMailboxDelegationRule(uid);
					mailboxDelegationRule.delegateUids.remove(groupUid);
					mailboxService.setMailboxDelegationRule(uid, mailboxDelegationRule);
				});
	}

	private List<String> getMailboxes(String domainUid, String groupUid) {
		return ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IDirectory.class, domainUid)
				.searchUids(DirEntryQuery.filterKind(Kind.SHARED_MAILBOX)).values.stream()//
				.filter(uid -> canAccessMailbox(groupUid, uid)).toList();
	}

	private boolean canAccessMailbox(String groupUid, String mailboxUid) {
		return ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IContainerManagement.class, "mailbox:acls-" + mailboxUid).getAccessControlList().stream()
				.anyMatch(ac -> ac.subject.equals(groupUid)
						&& (ac.verb.can(Verb.SendAs) || ac.verb.can(Verb.SendOnBehalf)));
	}

}
