/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.tx.outbox.persistence;

import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import javax.sql.DataSource;

import net.bluemind.core.jdbc.JdbcAbstractStore;
import net.bluemind.tx.outbox.repository.FlushSeqHolder;
import net.bluemind.tx.outbox.repository.ITxOutboxRepository;

public class TxOutboxStore extends JdbcAbstractStore implements ITxOutboxRepository {
	private final Supplier<FlushSeqHolder> seqHolder;

	public TxOutboxStore(DataSource dataSource) {
		super(dataSource);
		seqHolder = () -> FlushSeqHolder.get(dataSource);
	}

	public static final String INSERT = """
			INSERT INTO tx_outbox_for_kafka
			(domain_uid, part_key, kafka_key, kafka_value)
			VALUES
			(?, ?, ?, ?)
			RETURNING id
			""";

	public long forKafka(String domainUid, String partKey, byte[] key, byte[] value) throws SQLException {
		return unique(INSERT, rs -> rs.getLong(1), Collections.emptyList(),
				new Object[] { domainUid, partKey, key, value });
	}

	public static final String PENDING = """
			SELECT id FROM tx_outbox_for_kafka
			WHERE id != ALL(?)
			ORDER BY id ASC LIMIT ?
			""";

	public List<Long> kafkaPending(int limit) throws SQLException {
		return kafkaPending(limit, Collections.emptyList());
	}

	@Override
	public List<Long> kafkaPending(int limit, List<Long> inFlight) throws SQLException {
		return selectLong(PENDING, new Object[] { inFlight.stream().mapToLong(Number::longValue).toArray(), limit });
	}

	public static final String LOAD = """
			SELECT domain_uid, part_key, kafka_key, kafka_value
			FROM tx_outbox_for_kafka
			WHERE id = ?
			""";

	@Override
	public KafkaPayload get(long id) throws SQLException {
		return unique(LOAD,
				rs -> new KafkaPayload(id, rs.getString(1), rs.getString(2), rs.getBytes(3), rs.getBytes(4)),
				Collections.emptyList(), new Object[] { id });
	}

	public static final String M_LOAD = """
			SELECT domain_uid, part_key, kafka_key, kafka_value, id
			FROM tx_outbox_for_kafka
			WHERE id = ANY(?)
			""";

	@Override
	public List<KafkaPayload> mget(List<Long> ids) throws SQLException {
		return select(M_LOAD,
				rs -> new KafkaPayload(rs.getLong(5), rs.getString(1), rs.getString(2), rs.getBytes(3), rs.getBytes(4)),
				Collections.emptyList(), new Object[] { ids.stream().mapToLong(Number::longValue).toArray() });
	}

	public static final String DEL = """
			DELETE FROM tx_outbox_for_kafka
			WHERE id = ANY(?)
			""";

	@Override
	public void deleteOffsets(List<Long> id) throws SQLException {
		delete(DEL, new Object[] { id.stream().mapToLong(Number::longValue).toArray() });
	}

	@Override
	public FlushSeqHolder seqHolder() {
		return seqHolder.get();
	}

}
