/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.delivery.lmtp.dedup;

import static net.bluemind.delivery.lmtp.dedup.DuplicateDeliveryDb.UniqueMessagePostAction.NO_POST_ACTION;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.atomic.LongAdder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.hash.HashFunction;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.netflix.spectator.api.Counter;
import com.netflix.spectator.api.Registry;

import io.lettuce.core.RedisClient;
import io.lettuce.core.SetArgs;
import io.lettuce.core.api.StatefulRedisConnection;
import io.lettuce.core.api.async.RedisAsyncCommands;
import io.lettuce.core.api.sync.RedisCommands;
import io.lettuce.core.codec.ByteArrayCodec;
import net.bluemind.delivery.lmtp.common.FreezableDeliveryContent;
import net.bluemind.delivery.lmtp.common.ResolvedBox;
import net.bluemind.delivery.lmtp.config.DeliveryConfig;
import net.bluemind.keydb.common.ClientProvider;
import net.bluemind.metrics.registry.IdFactory;
import net.bluemind.metrics.registry.MetricsRegistry;

public class DuplicateDeliveryDb {

	@FunctionalInterface
	public interface UniqueMessageAction {

		void run() throws IOException;

	}

	@FunctionalInterface
	public interface UniqueMessagePostAction {

		static final UniqueMessagePostAction NO_POST_ACTION = () -> {
		};

		void run();

	}

	private static final Logger logger = LoggerFactory.getLogger(DuplicateDeliveryDb.class);

	private final Duration window;
	private final Counter dedupCounter;

	private final LongAdder deduplications;
	private final RedisCommands<byte[], byte[]> commands;
	private final RedisAsyncCommands<byte[], byte[]> pushCommands;
	private static final HashFunction hasherFunction = Hashing.murmur3_128();

	private static final byte[] CONST_VALUE = new byte[] { 0x01 };

	private static final DuplicateDeliveryDb INSTANCE = new DuplicateDeliveryDb(
			DeliveryConfig.get().getDuration("lmtp.dedup.window"));

	public static final DuplicateDeliveryDb get() {
		return INSTANCE;
	}

	public DuplicateDeliveryDb(Duration dedupWindow) {
		this.window = dedupWindow;
		this.deduplications = new LongAdder();
		Registry reg = MetricsRegistry.get();
		IdFactory idf = new IdFactory("bm-lmtpd", reg, DuplicateDeliveryDb.class);
		this.dedupCounter = reg.counter(idf.name("deduplicated"));

		RedisClient redisClient = ClientProvider.newClient();
		StatefulRedisConnection<byte[], byte[]> connection = redisClient.connect(ByteArrayCodec.INSTANCE);
		commands = connection.sync();
		pushCommands = connection.async();
		logger.info("Duplicate delivery protection started, duration ~ {}h.", window.toHours());
	}

	public long dedupCount() {
		// we can't use the spectator counter in tests as the implementation is a noop
		// when the java agent is not setup
		return deduplications.sum();
	}

	public boolean runIfUnique(FreezableDeliveryContent fc, UniqueMessageAction action) throws IOException {
		String messageId = fc.content().message().getMessageId();
		return (fc.content().message().getMessageId() != null) //
				? runIfUnique(messageId, getKey(messageId, fc.content().box(), fc.serializedMessage().guid()), action)
				: runAction(action, NO_POST_ACTION, NO_POST_ACTION);
	}

	public boolean runIfUnique(String messageId, byte[] keyBytes, UniqueMessageAction action) throws IOException {
		if (!exists(keyBytes)) {
			return runAction(action, () -> putOrFail(keyBytes), () -> deleteOrFail(keyBytes));
		} else {
			logger.warn("Message delivery {} skipped as message id was seen in the last {} day(s)", messageId,
					window.toDays());
			dedupCounter.increment();
			deduplications.increment();
			return false;
		}
	}

	protected boolean runAction(UniqueMessageAction action, UniqueMessagePostAction onSuccess,
			UniqueMessagePostAction onFailure) throws IOException {
		try {
			action.run();
			onSuccess.run();
			return true;
		} catch (IOException ioe) {
			onFailure.run();
			logger.error("runAction failure", ioe);
			throw ioe;
		} catch (Exception e) {
			onFailure.run();
			logger.error("runAction failure", e);
			throw new IOException(e);
		}
	}

	private byte[] getKey(String messageId, ResolvedBox target, String guid) {
		Hasher hasher = hasherFunction.newHasher();
		hasher.putString(messageId, StandardCharsets.UTF_8);
		hasher.putString(target.dom.uid, StandardCharsets.UTF_8);
		hasher.putString(target.entry.entryUid, StandardCharsets.UTF_8);
		hasher.putString(guid, StandardCharsets.UTF_8);
		return ("lmtp:" + hasher.hash().toString()).getBytes();
	}

	private void putOrFail(byte[] keyBytes) {
		pushCommands.set(keyBytes, CONST_VALUE, SetArgs.Builder.ex(window));
	}

	private boolean exists(byte[] keyBytes) {
		return commands.exists(keyBytes) > 0;
	}

	private void deleteOrFail(byte[] keyBytes) {
		pushCommands.del(keyBytes);
	}

}
