/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.indexing.incremental.mail;

import java.util.List;
import java.util.Optional;

import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.indexing.IMailIndexService;
import net.bluemind.backend.mail.replica.indexing.RecordIndexActivator;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.indexing.incremental.AbstractTypeIndexer;
import net.bluemind.indexing.incremental.AbstractTypeIndexer.IndexWriter;
import net.bluemind.indexing.incremental.NoopIndexer;
import net.bluemind.indexing.incremental.TypeIndexerFactory;

public class RecordsIncrementalIndexFactory implements TypeIndexerFactory {

	@Override
	public String type() {
		return IMailReplicaUids.MAILBOX_RECORDS;
	}

	@Override
	public TypeIndexer create(BmContext ctx, Container c) {
		String uniqueId = IMailReplicaUids.uniqueId(c.uid);
		try {
			IDbMailboxRecords recApi = ctx.provider().instance(IDbMailboxRecords.class, uniqueId);
			Optional<IMailIndexService> optIdx = RecordIndexActivator.getIndexer();
			return optIdx.<TypeIndexer>map(midx -> new RecIndexer(recApi, new RecIndexWriter(c.owner, uniqueId, midx)))
					.orElseGet(NoopIndexer::new);
		} catch (ServerFault sf) {
			// we skip on "not found", the folder delete will be processed
			// as a subtree change
			if (sf.getCode() == ErrorCode.NOT_FOUND) {
				return new NoopIndexer();
			}
			throw sf;
		}
	}

	public static class RecIndexWriter implements IndexWriter<MailboxRecord> {

		private IMailIndexService mailIndex;
		private String owner;
		private String uniqueId;

		public RecIndexWriter(String owner, String uniqueId, IMailIndexService midx) {
			this.mailIndex = midx;
			this.uniqueId = uniqueId;
			this.owner = owner;
		}

		@Override
		public void write(ItemValue<MailboxRecord> toIndex) {
			mailIndex.storeMessage(uniqueId, toIndex, owner);
		}

		@Override
		public void deleteByIds(List<Long> itemIds) {
			mailIndex.expunge(owner, uniqueId, itemIds);
		}

	}

	public static class RecIndexer extends AbstractTypeIndexer<MailboxRecord> {

		public RecIndexer(IDbMailboxRecords rec, IndexWriter<MailboxRecord> writer) {
			super(rec, rec, writer);
		}

	}

}
